#include <stdio.h>
#define YEAR 2011
#define MAX_YEARS 12

typedef struct _country {
	char name[16];
	float debt_on_gpd; /* debt/gpd % */
	float	gpd_growth; /* gpd[i+1]/gpd[i] % */
	float gpd_pro_capite;  /* in $ */
	float balance;  /* *10^9 $ */
	float expenses; /* *10^9 $ */
	float revenues; /* *10^9 $ */
	float inflation; /* % */
	float govt_bond_rates; /* value/nominal % */
	float population; /* *10^6 */
} country;

/* Read a list of countries from stdin, skipping the first line of input
 * Parameters:
 *     europe		The array of countries
 * Returns: the number of country entries read
 */
int read_countries(country europe[]){
	int res=0, i=0;
	char buffer[128];
	fgets(buffer,127,stdin); /* reads and ignores the first line of input */
	printf("%s",buffer);
	do {
		/* res=read_country(europe[i]); */
		res=scanf("%s %f %f %f %f %f %f %f %f %f\n", 	
		europe[i].name,
		&(europe[i].debt_on_gpd), &(europe[i].gpd_growth), 
		&(europe[i].gpd_pro_capite), &(europe[i].balance), 
		&(europe[i].expenses), &(europe[i].revenues), 
		&(europe[i].inflation), &(europe[i].govt_bond_rates), 
		&(europe[i].population));
		i++;
	} while(res>=0);
	return i;
}

/* Print the name of each country in the array */
void print_country_names(country europe[], int ncountries){
	int i;
	printf("Countries (%d): ", ncountries);
	for (i=0; i<ncountries; i++)
		printf("%s ", europe[i].name);
	printf("\n");
}

void print_country(country the_country){
	printf("%s %0.2f %0.2f %0.2f %0.2f %0.2f %0.2f %0.2f %0.2f %0.2f\n", 	
		the_country.name,	
		the_country.debt_on_gpd, the_country.gpd_growth, 
		the_country.gpd_pro_capite, the_country.balance, 
		the_country.expenses, the_country.revenues, 
		the_country.inflation, the_country.govt_bond_rates, 
		the_country.population);
}

/* Initialize the name of all countries in the array to an empty string */
void init_countries(country europe[], int ncountries){
	int i;
	for (i=0; i<ncountries; i++)
		europe[i].name[0]='\0';
}

/* Compute debt progression for several years 
 * Parameters:
 *    the_country	The country 
 *    debt		An array to hold the debt forecast for the_country
 *    years		The number of elements of the debt array
 * Returns: nothing, but the result is written to debt
 */
void compute_debt(country the_country, float debt[], int years){
	float gpd = the_country.gpd_pro_capite * 
			the_country.population;
	float deficit_on_gpd = (the_country.expenses - 
				the_country.revenues)/gpd;
	float rate_to_growth_ratio = 
		((100+the_country.govt_bond_rates)/
		(100+the_country.gpd_growth));
	int i;
	debt[0]=the_country.debt_on_gpd;
	for(i=1; i<years; i++){
		debt[i]=debt[i-1] * rate_to_growth_ratio + 
			deficit_on_gpd;
	}
}

/* Print approximate debt/gpd for several years, on a 70*24 console display 
 * 1 character = 10% -> height of '*' = (int)(debt/10)
 * Parameters:
 *    the_country	The country 
 *    debt		An array holding the debt forecast for the_country
 *    years		The number of elements of the debt array
 */
void print_debt(country the_country, float debt[], int years){
	int height[MAX_YEARS];
	int i,j;
	printf("%s debt/gpd forecast [%%]\n", the_country.name);
	if (years>MAX_YEARS) years=MAX_YEARS; /* A limit to avoid printing out of line */
	for(i=0; i<years; i++) {
		height[i]= (int) (debt[i]/10);
	}
	for(i=19;i>=0;i--) {
		for(j=0; j<years; j++)
			if (height[j]==i) printf("*****");
			else printf("     ");
		printf("\t%d%%\n", i*10);
	}
	for(i=0;i<years;i++)
		printf(" %4d",i+YEAR);
	printf("\n");
}

/* Search a country by name in the country list
 * Parameters:
 *    countries		The array of countries
 *    ncountries	The number of valid elements of countries
 *    name		The name of the country to be found
 * Returns: the index of the country entry in the array
 */
int lookup(country countries[], int ncountries, char name[]){
	int i;
	for(i=0; i<ncountries; i++){
		if (strcmp(countries[i].name, name)==0)
			return i;
	}
	return -1; /* Not found */
}

int main(int argc, char **argv){
	country europe[16];
	float debt_per_year[10];
	int ncountries=0;
	int selection=-1;
	int nyears=10;
	
	/* Initialization */
	init_countries(europe,16);

	/* Input acquisition */
	ncountries=read_countries(europe);

	/* Input acquisition checks */
	print_country_names(europe,ncountries);

	if (argc<2) return 0;

	selection=lookup(europe, ncountries, argv[1]);
	if (selection==-1) return 1;
	
	print_country(europe[selection]);

	/* Program logic  */
	compute_debt(europe[selection],debt_per_year,nyears);

	/* Output generation */
	print_debt(europe[selection],debt_per_year,nyears);

}
