#include "level.h"
#include <stdio.h>
#include <string.h>
#include "utils.h"
#include "printer.h"
#include <math.h>

Level *load_level_map(char *filename){
	int i, nchars;
	FILE *fin=open_file(filename,"r");
	Level *level=dmalloc(sizeof(Level));

	/* Read n cols and n rows from file */
	fscanf(fin,"%d\n%d\n", &(level->rows), &(level->cols));
	level->map=dmalloc(sizeof(char *)*level->rows);
	level->explored=dmalloc(sizeof(char *)*level->rows);
	
	for(i=0; i<level->rows; i++){
		level->map[i]=dmalloc(sizeof(char *)*level->cols);
		level->explored[i]=dmalloc(sizeof(char *)*level->cols); 
		/*! Note: explored is automatically initialized to 0 by dmalloc */
		nchars=fscanf(fin,"%s\n", level->map[i]);
		if(nchars<1 || strlen(level->map[i])<level->cols) {
			fprintf(stderr,"Read error at line %d (rows: %d, cols: %d): %s\n", i, 
							level->rows, level->cols, level->map[i]);
			fprintf(stderr,"Items read: %d; string length: %d\n", nchars, 
							(int)strlen(level->map[i]));
			return NULL;		
		}
		
	}
	return level;
}

Level *load_level(char *filename){
	int i, depth, end=1;
	char buf1[128];
	char buf2[128];
	Level *level=NULL;
	FILE *fin=open_file(filename, "r");

	fscanf(fin,"name:%s\n",buf1);
	fscanf(fin,"depth:%d\n",&depth);
	fscanf(fin,"map:%s\n", buf2);
	level=load_level_map(buf2);
	
	level->depth=depth;
	level->objs=dmalloc(sizeof(content)*MAXITEMS);
	reset(level->objs,sizeof(content)*MAXITEMS);
	
	for(i=0; end!=EOF && i<MAXITEMS; i++){
		int x, y, n, m, s;
		char sym;
		unsigned itype;
		end=fscanf(fin,"[%d,%d]:%c ", &x, &y, &sym);
		if (end<3) continue;
		debug("reading symbol %c", sym);
		itype=get_item_type_from_symbol(sym);
		if (itype!=NOITEM) {
			end=fscanf(fin,"%d %d %d\n", &n, &m, &s);
			add_item_to_level(level, x, y, itype, n, m, NULL, s);
		} else {
			level->objs[i].stash=NULL;
		}
		itype=get_monster_type_from_symbol(sym);
		if (itype!=NOITEM) {
			end=fscanf(fin,"\n");
			add_monster_to_level(level, x, y, itype);
		} else {
			level->objs[i].npc=NULL;
		}
		level->objs[i].special=0;
	}
	level->objs[i].valid=0;
	
	return level;
}

void print_level(Level *level, int row, int col, char pcsymbol){
	char **map;
	int i, j;
	/*! Create memory for the map and copy the base map from level */
	map=dmalloc(sizeof(char *)*level->rows);
	for(i=0; i<level->rows; i++){
		map[i]=strdup(level->map[i]);
	}
	/*! Add symbols for objects and monsters 
	 * \todo Monsters should be added after items, actually.
	 */
	for(i=0; level->objs[i].valid!=0 && i<MAXITEMS; i++){
		int x, y;
		x=level->objs[i].row;
		y=level->objs[i].col;
		if (level->objs[i].valid==VALID && line_of_sight(level,row,col,x,y)) {
			if (level->objs[i].stash) map[x][y]=get_item_symbol(level->objs[i].stash);
			if (level->objs[i].npc) map[x][y]=level->objs[i].npc->species->symbol;
		}
	}
	/*! Add the symbol corresponding to the PC */
	map[row][col]=pcsymbol;
	
	/*! Blank out unexplored regions */
	for(i=0; i<level->rows; i++)
		for(j=0; j<level->cols; j++){
			if (!level->explored[i][j]) map[i][j]=' ';
		}
	
	/*! Print out the map using the printer.h facilities */
	print_map(map, level->rows);
	
	/*! Deallocate the memory used for the map */
	for(i=0; i<level->rows; i++){
		dfree(map[i]);
	}
	dfree(map);
}

int count_objs_in_level(Level *level){
	int i;
	for(i=0; level->objs[i].valid!=INVALID && i<MAXITEMS; i++);
	return i;
}

void add_item_to_level(Level *level, int row, int col, item_types itype,
											 int quantity, int magic, char *text, int special){
		int i = count_objs_in_level(level);
		level->objs[i].row=row;
		level->objs[i].col=col;
		level->objs[i].stash=create_item(itype, quantity, magic, text, special);
		level->objs[i].valid=VALID;
}

void *remove_item_from_level(Level *level, int i) {
	void *res=NULL;
	level->objs[i].valid=REMOVED;
	if (level->objs[i].stash) res=level->objs[i].stash;
	if (level->objs[i].npc) res=level->objs[i].npc;
	level->objs[i].npc=NULL;
	level->objs[i].stash=NULL;
	return res;
}

monster *find_monster_at(Level *level, int row, int col){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].valid==VALID && level->objs[i].row==row && 
		    level->objs[i].col==col && level->objs[i].npc!=NULL)
			return level->objs[i].npc;
	return NULL;
}

content *find_monster_pos(Level *level, monster *m){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].valid==VALID && level->objs[i].npc==m)
			return &(level->objs[i]);
	return NULL;	
}

monster *find_monster_adjacent_to(Level *level, int pc_row, int pc_col){
	static direction dir;
	static int row;
	static int col;
	monster *res;
	/* Initialization */
	if (pc_row>=0 && pc_col>=0) {
		row=pc_row;
		col=pc_col;
		dir=NORTH;
	} 
	/* General operation */
	res=NULL;
	switch(dir){
		case NORTH:      res=find_monster_at(level, row-1,col  ); dir++; if (res) return res;
		case NORTHEAST : res=find_monster_at(level, row-1,col+1); dir++; if (res) return res;
		case EAST :      res=find_monster_at(level, row  ,col+1); dir++; if (res) return res;
		case SOUTHEAST : res=find_monster_at(level, row+1,col+1); dir++; if (res) return res;
		case SOUTH :     res=find_monster_at(level, row+1,col  ); dir++; if (res) return res;
		case SOUTHWEST : res=find_monster_at(level, row+1,col-1); dir++; if (res) return res;
		case WEST :      res=find_monster_at(level, row  ,col-1); dir++; if (res) return res;
		case NORTHWEST : res=find_monster_at(level, row-1,col-1); dir++; if (res) return res;
		default : return NULL;
	}
}

monster *get_next_active_monster(Level *level){
	static int i;
	static Level *l;
	monster *res;
	if (level!=NULL) {
		l=level;
		i=0;
	}
	while(i<count_objs_in_level(l))
		if (l->objs[i].valid==VALID && l->objs[i].npc!=NULL && l->objs[i].npc->turn==0) {
			i++;
			return l->objs[i-1].npc;
		} else {
			i++;
		}
	/*! No more monsters to return */
	return NULL;
}

void remove_monster_from_level(Level *level, monster *npc){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].npc == npc) {
			destroy_monster(level->objs[i].npc);
			level->objs[i].valid=REMOVED;
		}		 
}

void add_monster_to_level(Level *level, int row, int col, monsters type){
	int i = count_objs_in_level(level);
	level->objs[i].row=row;
	level->objs[i].col=col;
	level->objs[i].npc=create_monster(type);
	level->objs[i].valid=VALID;
}

occupation can_move_to(Level *level, int row, int col){
	if (level->map[row][col]!='.') return BUSY;
	if (find_monster_at(level, row, col)) return ATCK;
	return FREE;
}


int block_los(Level *level, int row, int col){
	switch (level->map[row][col]) {
		case '#' : /* Add all other opaque blocks here */
		case '8' : return TRUE;
		default : return FALSE;
	}
}

/*! \brief Direction from (x0, y0) towards a given position (x1, y1) */
direction dir_to(int x0, int y0, int x1, int y1){
	int dx, dy, sx, sy;
	dx = x1-x0;
	dy = y1-y0;
	if (dy>0 && dx==0) return NORTH;
	else if (dy>0 && dx>0) return NORTHEAST;
	else if (dy==0 && dx>0) return EAST;
	else if (dy<0 && dx>0) return SOUTHEAST;
	else if (dx==0 && dy<0) return SOUTH;
	else if (dy<0 && dx<0) return SOUTHWEST;
	else if (dy==0 && dx<0) return WEST;
	else if (dy>0 && dx<0) return NORTHWEST;
	/*! TODO */
	return INVDIR; /* Some error... */
}

int line_of_sight(Level *level, int a_row, int a_col, int b_row, int b_col){
	/*! Implementation based on Wikipedia's version of Bresenham's line algorithm 
	 *  http://en.wikipedia.org/wiki/Bresenham's_line_algorithm
	 */
	int dx, dy, sx, sy, x0, x1, y0, y1, err, e2;
	x0=a_row;
	x1=b_row;
	y0=a_col;
	y1=b_col;

	/* Using the three operand expression of C, which evaluates to the second
	 * operand is the first is true, or to the third operand otherwise  */
	dx = (x1>x0 ? x1-x0 : x0-x1);
	dy = (y1>y0 ? y1-y0 : y0-y1);
	sx = (x0<x1 ? 1 : -1);
	sy = (y0<y1 ? 1 : -1);
	err = dx-dy;

  if (x0==x1 && y0==y1) return TRUE;
	while(1){ 
     e2=2*err;
     if (e2>-dy) {
       err-=dy;
       x0+=sx;
     }
     if (x0==x1 && y0==y1) return TRUE;
     if (e2 <  dx) {
       err+=dx;
       y0+=sy;
     }
     if (x0==x1 && y0==y1) return TRUE;
     if (block_los(level, x0, y0)) return FALSE;
	}

	printf("We've got a problem here...\n");
	return TRUE;
}

int monster_LoS_to(Level *level, monster *m, int row, int col){
	content *pos=find_monster_pos(level, m);
	if (pos==NULL){
		print_msg("Uh oh, I seem to have lost track of a %s...", m->species->name);
		return;
	}
	return line_of_sight(level, pos->row, pos->col, row, col);
}

void mark_explored(Level *level, int row, int col){
	int i, j;
	/*! Blank out unexplored regions */
	level->explored[row][col]=TRUE;
	for(i=0; i<level->rows; i++)
		for(j=0; j<level->cols; j++){
			if (line_of_sight(level, row, col, i, j)) level->explored[i][j]=TRUE;
		}
}

void move_monster_dir(Level *level, monster *m, direction dir){
	int x0, y0, x1, y1;
	content *pos=find_monster_pos(level, m);
	if (pos==NULL){
		print_msg("Uh oh, I seem to have lost track of a %s...", m->species->name);
		return;
	}
	x0=pos->row;
	y0=pos->col;
	switch(dir){
		case NORTH:      x1=x0; y1=y0+1; break;
		case NORTHEAST : x1=x0+1; y1=y0+1; break;
		case EAST :      x1=x0+1; y1=y0; break;
		case SOUTHEAST : x1=x0+1; y1=y0-1; break;
		case SOUTH :     x1=x0; y1=y0-1; break;
		case SOUTHWEST : x1=x0-1; y1=y0-1; break;
		case WEST :      x1=x0-1; y1=y0; break;
		case NORTHWEST : x1=x0-1; y1=y0+1; break;
		default : break;
	}
	if (can_move_to(level, x1, y1)==FREE) {
		pos->row=x1;
		pos->col=y1;
	}	
}

void move_monster_random(Level *level, monster *m){
	direction dir=randint(NORTH, NORTHWEST);
	move_monster_dir(level, m, dir);
}

/*! \brief Get the monster position and find the direction towards a goal */
direction monster_dir_to(Level *level, monster *m, int row, int col){
	content *pos=find_monster_pos(level, m);
	if (pos==NULL){
		print_msg("Uh oh, I seem to have lost track of a %s...", m->species->name);
		return;
	}
	return dir_to(pos->row, pos->col, row, col);
}

int monster_distance(Level *level, monster *m, int row, int col){
	content *pos=find_monster_pos(level, m);
	int dx, dy;
	if (pos==NULL){
		print_msg("Uh oh, I seem to have lost track of a %s...", m->species->name);
		return;
	}
	dx=pos->col-col;
	dx=dx*dx;
	dy=pos->row-row;
	dy=dy*dy;
	return (int) ceil(sqrt(dx+dy));
}

/*! \brief Reverse of a direction 
 *  \param dir The direction of which we need to know the opposite 
 *  \return The opposite direction, computed as ((dir-NORTH)+4)%8+NORTH
 *  This is correct because:
 *   - Directions are consecutive
 *   - INVDIR = 0
 *   - So we remove INVDIR from the count, rebasing dir to 0.
 *   - Then the opposite to each direction is dir+4, computed modulo the number
 *     of directions
 *   - Finally, we need to rebase back to NORTH
 */
direction reverse_dir(direction dir){
	return NORTH+(dir+3)%8;
}

void monster_flee(Level *level, monster *m, int row, int col){
	direction dir = monster_dir_to(level, m, row, col);
	dir=reverse_dir(dir);
	move_monster_dir(level, m, dir);
}

void monster_goto(Level *level, monster *m, int row, int col){
	direction dir = monster_dir_to(level, m, row, col);
	move_monster_dir(level, m, dir);
}


void reset_monster_turns(Level *level){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].valid==VALID && level->objs[i].npc!=NULL)
			level->objs[i].npc->turn=0;
}


#ifdef TEST_LEVEL
void print_end(char *s) {
	#ifdef CURSES
		refresh();
		napms(1000);
		mvprintw(LINES-1, COLS-11,s); 
		refresh();
		napms(1000);
	#endif /* CURSES */
}

int main(int argc, char **argv){
	Level *level;
	
	screen_init();
	level=load_level(argv[1]);
	mark_explored(level,8,2);
	print_level(level,8,2,'@');
	print_end("Curses Test");
	add_item_to_level(level, 7, 4, SCROLL, 1, 1, 0, 0);
	print_level(level,8,2,'@');
	print_end("%% Added    "); 
	screen_close();
	
	#ifdef CURSES
		printf("Test finished with curses\n");
	#else
		printf("Test finished without curses\n");
	#endif /* CURSES */
}
#endif /* TEST_LEVEL */
