#ifndef MONSTERS_H
#define MONSTERS_H
/*! \file monsters.h
 *  \brief Description of monsters and their types
 */
 
 #include "character.h"
 
 /*! \brief Enumerator of the monster types */
 typedef enum _monsters {
 	NOMONST,
	RAT,
	KOBOLD,
	GOBLIN,
	WOLF,
	WYVERN,
	DRAGON,
	NMONSTERTYPES
 } monsters;
 
 /*! \brief Monster type 
  *
  * This data structure contains information that are common to all
	* monsters of the same type. 
  */
typedef struct _monster_type {
	monsters type;
	char symbol; /*! Could be replaced with an enum type */	
	char *name; /*! The monster type name */
	short int might; /*! Might Attack/Defense */
	short int speed; /*! Speed Attack/Defense */
	short int mind;  /*! Mind  Attack/Defense */
	short int hits;  /*! Max hit points */
	short int dmg;   /*! Damage per attack */
} monster_type;

/*! \brief Structure for individual monsters 
 */
typedef struct _monster {
	short int hits; /*! when this gets to zero, the monster dies */
	monster_type *species; /*! A pointer to the monster type structure */
	short int turn; /*! set to 1 after the monster has acted for its turn */
} monster;


/*! \brief Enumeration type for possible monster actions 
 * \todo In the future, monsters may have other actions, such as speaking
 *       or bartering with the PC, attacking monsters of different factions,
 *       or even allying with the PC, etc.
 */
typedef enum _monster_action {
  MONSTER_NO_ACTION, /*! No action, without rest effects (turn lost) */
	MONSTER_REST, /*! Monster will do nothing (although rest effects are possible) */
	MONSTER_ATTACK, /*! Monster will attack the PC with a melee action */
	MONSTER_FIRE, /*! Monster will attack the PC with a ranged weapon */
	MONSTER_CAST, /*! Monster will attack the PC with a spell */
	MONSTER_MOVE, /*! Monster will move randomly */
	MONSTER_FLEE, /*! Monster will flee from the PC */
	MONSTER_HUNT  /*! Monster will follow the PC */
} monster_action;


/*! \brief Create a monster of the given type */
monster *create_monster(monsters type);

/*! \brief Deallocate a monster of the given type */
void destroy_monster(monster *m);

/*! \brief Get the monster type from the symbol (char) */
monsters get_monster_type_from_symbol(char sym);

/*! \brief Monster attack against PC */
void monster_attack(monster *m, character *pc);

/*! \brief Compute XPs per monster */
int monster_xp(monster *m, character *pc);

/*! \brief Decide course of action for this turn 
 * \param m A monster
 * \param pc The player character, or NULL if the PC is not in sight
 * \param distance The distance between the monster and the PC 
 *        (non-significant if pc==NULL)
 */
monster_action monster_ai(monster *m, character *pc, int distance);

 #endif /* MONSTERS_H */
