#ifndef ITEMS_H
#define ITEMS_H
/*! \file items.h
 *  \brief Definition of items
 */
#define TRUE 1
#define FALSE 0

/*! \brief Enumeration of possible item types
 */
typedef enum _item_types {
	NOITEM,
	COINS,
	TREE,
	FUNGUS,
	ARROW,
	SCROLL,
	SWORD,
	BOW,
	TOME,
	NITEMTYPES
} item_types;

/*! \brief Structure describing an individual item
 *
 * Items that can be inserted in the inventory 
 * Actions PICK & DROP are available for all objects
 * This structure only represents properties of the 
 * individual item, not of a class of items
 */
typedef struct _item {
	int quantity; /*! Number of identical objects in the group */
	item_types type; /*! Index of the object type. 
	                  * Exercise: replace with pointer*/
	int magic;   /*! item magic/quality bonus */
	char *text;  /*! item can be read, with associated text */ 
	int special; /*! special effect; depends on type */
} item;

/*! \brief Structure defining an item class
 *
 *  This class specifies which actions can be performed on each 
 *  object class: EAT, WIELD, QUIVER
 */
typedef struct _item_class {
	item_types type; /*! Enumerator of the object class           */
	char symbol;/*! The character used to represent these objects */
	char *name; /*! String correspoding to the object class name  */
	int maxn;   /*! maximum occurrences in a group of these items */
	int magic;  /*! 1: item can be magic; 0: item cannot be magic */
	int food;   /*! item can be used as food, with that value     */
	int damage; /*! base damage (for weapons and tomes)           */
	int wieldable; /*! can be used as weapon/spell */ 
	int quiver; /*! can be used as ammunition for bows */
} item_class;

/*! \brief The actual list of available object types in the game */
extern item_class item_classes[];

/*! \brief Return the object type corresponding to a symbol */
item_types get_type_from_symbol(char sym);

/*! \brief Get the symbol corresponding to an item */
char get_item_symbol(item *it);

/*! \brief Create the data structure for a specific object */
item *create_item(item_types type, int quantity, 
          int magic, char *text, int special);

/*! \brief Return the damage inflicted by a weapon/spell 
 *  
 * The damage inflicted by an object is computed as the sum 
 * of the base damage of the object class and the magic bonus
 * of the specific item.
 * \returns A positive or 0 integer which represents the damage
 */
int damage(item *it);

#endif /* ITEMS_H */
