#ifndef LEVEL_H
#define LEVEL_H
/*! \file level.h
 *  \brief Dungeon level description and content
 */
#include "monsters.h"
#include "direction.h"
#include "list.h"

/* Forward declarations */
struct _Level;

/*! The maximum number of distinct entries in the content array */
#define MAXITEMS 100

/*! Status of a map cell */
typedef enum _occupation {
	BUSY /*! The cell is permanently occupied by a wall or such */, 
	FREE /*! The cell is free (though items can be there) */,
	ATCK /*! The cell is occupied by a monster. Moving there means to attack */,
	STAIRS_UP /*! THe cell contains stairs to an upper level */,
	STAIRS_DOWN /*! The cells contains stairs to a lower level */
} occupation;

/*! \brief Status of an object in the level object array. 
 *
 * This enum is used because the set of objects in the level is defined
 * as a (dynamic) array of content structures. 
 * \todo It would be better handled as a list (implementing this feature is 
 * a good exercise for the winter break!)
 */
typedef enum _obj_valid {
	INVALID /*! Not used */,
	VALID /*! The object is valid */,
	REMOVED /*! The object was destroyed or removed */
} obj_valid;

/*! This is a placeholder for future development
 *  Effects are merely numbered, for now */
typedef enum _effect { NOEFFECT } effect;

/*! \brief This structure represents the content of a dungeon cell 
 *
 * There may be multiple object stashes or stashes and a monster
 * in a single cell
 * \todo Exercise: implement a check to verify that only one monster is present.
 * \todo Exercise: merge stashes in the same location, but not those associated
 * with an eventual monster.
 */
typedef struct _content{
	int row;         /*! position: row */ 
	int col;         /*! position: col */
	item *stash;     /*! Items present in the cell
	                  *  if monster!=NOMONSTER, these items are moved with the monster 
	                  *  and part of its equipment/treasure */
	effect special;  /*! trap or special effect */
	monster *npc;    /*! monster or npc */
	obj_valid valid; /*! Whether the object is in use, removed, or not initialized */
	struct _Level *stairs_to; /*! Pointer to lower/upper levels */
} content;

/*! \brief Definition of a dungeon level 
 *
 * This information is read from a level configuration file
 */
typedef struct _Level {
	char *name;/*! Dungeon level name (e.g.: Temple of Orcus) */
	char *filename;/*! Dungeon level file name (i.e., unique ID) */
	char **map;/*! Array of arrays representing the level map */
	char **explored;/*! Array of arrays representing the level exploration map */
	int rows;  /*! Number of rows in map */
	int cols;  /*! Number of cols in map */
	int depth; /* Determines random threats levels */
	content *objs; /*! Array of content objects. 
	                *  \todo See \see obj_valid for possible variations */
} Level;

/*! \brief Load a level map from the specified file 
 *  \param filename  A file name, read from the level configuration file 
 *  \return   The level map 
 */
Level *load_level_map(char *filename);

/*! \brief Load a level from the specified file
 *  \param levels    The current list of levels
 *  \param filename  The file containing the description of the level
 *  \return   A list of Level data structures 
 */
elem *load_level(elem *levels, char *filename);

/*! \brief A function used to search levels by name */
int checkname(Level *level, char *filename);

/*! \brief A function to compare levels by depth */
int cmp_level_depth(Level *l1, Level *l2);

/*! \brief Adds a set of stairs to the level */
void add_stairs_to_level(Level *level, int row, int col, char sym, Level *dest);

/*! \brief Counts how many objects are found in the level */
int count_objs_in_level(Level *level);

/*! \brief Adds an item to the level */
void add_item_to_level(Level *level, int row, int col, item_types itype,
											 int quantity, int magic, char *text, int special);

/*! \brief Removes i-th object from the level */
void *remove_object_from_level(Level *level, int i);

/*! \brief Find if a position (row,col) is occupied by walls or
 * other impassable obstacles, or by a monster 
 */
occupation can_move_to(Level *level, int row, int col);

/*! \brief Print the level map using the printer library */
void print_level(Level *level, int row, int col, char pcsymbol);

/*! \brief Adds a new monster to the level at position row, col */
void add_monster_to_level(Level *level, int row, int col, monsters type);

/*! \brief Removes a monster from the level */
void remove_monster_from_level(Level *level, monster *npc);

/*! Determines if there is a monster at a given location (row, col) */
monster *find_monster_at(Level *level, int row, int col);

/*! Determines which level is entered from a given position (row, col) */
Level *get_level_entered_at(Level *level, int row, int col);

/*! \brief Update the exploration matrix to reflect what the PC can see from
 *   his/her position at (row,col) 
 */
void mark_explored(Level *level, int row, int col);

/*! \brief Check whether there is line of sight between two positions
 * \param level The current dungeon level
 * \param a_row Row for position a
 * \param a_col Col for position a
 * \param b_row Row for position b
 * \param b_col Col for position b
 * \return TRUE if there is line of sight, FALSE if sight is blocked
 */
int line_of_sight(Level *level, int a_row, int a_col, int b_row, int b_col);

#endif /*  LEVEL_H */
