#include "actions.h"
#include "printer.h"
#include <stdlib.h>

/*! \brief Move position according to direction command */
void move_pos(int *row, int *col, command dir);

/*! \brief Implements the NOCMD action; also serves as default action */
Level * action_none(Level *level, character *pc);

/*! \brief Implements all the move actions, includes check for occupied space */
Level * action_move(Level *level, character *pc, command cmd);

/*! \brief Implements the melee attack action */
Level * action_attack(Level *level, character *pc);

/*! \brief Implements the melee attack action, given the direction of the attack */
Level * action_attack_direction(Level *level, character *pc, command dir);

Level * action(Level *level, character *pc, command cmd){
	switch (cmd) {
		case NOCMD:   return action_none(level, pc); 
		case UP:	    /* All move actions implemented by the same function */
		case DOWN:    
		case LEFT:    
		case RIGHT:   
		case UPLEFT:  
		case UPRIGHT: 
		case DWLEFT:  
		case DWRIGHT: return action_move(level, pc, cmd); 
		case PICK:    /*! \todo PICK not yet implemented */
		case DROP:    /*! \todo DROP not yet implemented */ return action_none(level, pc); 
		case ATTACK:  return action_attack(level, pc); 
		case FIRE:    /*! \todo FIRE not yet implemented */
		case CAST:    /*! \todo CAST not yet implemented */
		case EAT:     /*! \todo EAT not yet implemented */
		case READ:    /*! \todo READ not yet implemented */
		case USE:     /*! \todo USE not yet implemented */
		case SAVE:    /*! \todo SAVE not yet implemented */
		case REST:    /*! \todo REST not yet implemented */
		case QUIVER:  /*! \todo QUIVER not yet implemented */
		case WIELD:   /*! \todo WIELD not yet implemented */
		default : return action_none(level, pc);
	}
	/*! This should be unreachable... */
	print_msg("Something wrong here... actions should never be unmanaged by dispatcher.");
	return level;
}

Level * action_none(Level *level, character *pc){
	/* empty on purpose */
	return level;
}

Level * action_move(Level *level, character *pc, command cmd){
	int row=pc->row;
	int col=pc->col;
	Level *new_level;
	occupation state;
	move_pos(&row,&col,cmd);
	state = can_move_to(level, row, col);
	switch (state) {
		case STAIRS_UP :
		case STAIRS_DOWN: 
		/*! Action for stairs (up or down are essentially the same)
		 *  - Ask confirmation to move to other level
		 *  - Move to other level or do nothing.
		 */ 
			new_level=get_level_entered_at(level, row, col);
			if (new_level==NULL) { 
				print_msg("Sorry, something not right here... there is no destination level!");
				break;
			}
			if (new_level==(Level *)0x1) { 
				print_msg("These stairs lead outside the dungeon!");
			} else {
				print_msg("These stairs lead to the %s level!", new_level->name);
			}
			print_msg("Do you want to proceed? [y/n]");
			if (get_yes_no()) {
				if (new_level==(Level *)0x1) { 
					print_msg("You have survived the dungeon! Congratulations. Your final score is %d xp.", pc->xp);
					get_cmd();
					screen_close();
					exit(0);
				} else {
					return new_level;
				}			
			}
		case BUSY: print_msg("Cannot move there..."); break;
		case FREE: pc->row=row; pc->col=col; break;
		case ATCK: action_attack_direction(level, pc, cmd); break;
	}
	return level;
}

/*! \brief Critical hit rule: if roll >= 16, add more damage */
int critical_damage(int roll){
	switch (roll) {
		case 20 : return 4;
		case 19 : return 3;
		case 18 : return 2;
		case 17 : return 1;
		case 16 : return 1;
		default : return 0;
	}
}

Level * action_attack_direction(Level *level, character *pc, command dir){
	monster *enemy=NULL;
	int roll=0;
	int erow=pc->row;
	int ecol=pc->col;
	int bonus=character_melee_attack_bonus(pc);
	move_pos(&erow,&ecol,dir);
	enemy=find_monster_at(level, erow, ecol);
	roll=d20();
	if (roll+bonus >= enemy->species->speed) {
		int dmg = character_melee_damage_bonus(pc) + 1 + critical_damage(roll); /*! \todo Only unarmed damage is supported right now */
		enemy->hits-=dmg;
		if (enemy->hits<=0) {
			print_msg("You hit the %s for %d damage, killing it!", enemy->species->name, dmg);
			/*! \todo Drop treasure from dead monster */
			/*! \todo Add XP for dead monster */
			pc->xp+=monster_xp(enemy,pc);
			remove_monster_from_level(level, enemy);
		} else
			print_msg("You hit the %s for %d damage", enemy->species->name, dmg);
	} else {
		print_msg("You missed!");
	}
	return level;
}

void move_pos(int *row, int *col, command dir){
	switch (dir) {
		case UP:	    *row-=1; break;
		case DOWN:    *row+=1; break;
		case LEFT:    *col-=1; break;
		case RIGHT:   *col+=1; break;
		case UPLEFT:  *row-=1; *col-=1; break;
		case UPRIGHT: *row-=1; *col+=1; break;
		case DWLEFT:  *row+=1; *col-=1; break;
		case DWRIGHT: *row+=1; *col+=1; break;
	}	
}

Level * action_attack(Level *level, character *pc){
	return action_attack_direction(level, pc, get_move_command());
}

