#ifndef ENEMY_ACTIONS_H
#define ENEMY_ACTIONS_H
/* \file enemy_actions.h
 * \brief Actions taken by monsters in the game
 */
#include "level.h"

/*! \brief Find and return monsters adjacent to (row, col)
 *  The function can be invoked multiple times, using a negative row and/or
 *  col for each invocation after the first. It will return, in time, all
 *  adjacent monsters, if present, until a NULL is returned to signal the
 *  end of the adjacent monsters.
 *  \param level   The current level
 *  \param pc_row  The horizontal position of the PC, or -1 after the first invocation
 *  \param pc_col  The vertical position of the PC, or -1 after the first invocation
 *  \return        One of the adjacent monsters, or NULL is none remain to return
 * 
 *  Note that this function employs static variables!
 */
monster *find_monster_adjacent_to(Level *level, int pc_row, int pc_col);

/*! \brief Move monster in a random direction 
 *  \param level The current level
 *  \param m The monster to move
 */
void move_monster_random(Level *level, monster *m);

/*! \brief Move monster in a given direction 
 *  \param level The current level
 *  \param m The monster to move
 *  \param dir The direction in which the monster moves
 */
void move_monster_dir(Level *level, monster *m, direction dir);

/*! \brief Move a monster far from a given position */
void monster_flee(Level *level, monster *m, int row, int col);

/*! \brief Move a monster towards a given position */
void monster_goto(Level *level, monster *m, int row, int col);

/*! \brief Reset all monster to turn==0 (not acted yet for this turn) */
void reset_monster_turns(Level *level);

/*! \brief Iterator over the monsters that have not already acted in a turn */
monster *get_next_active_monster(Level *level);

/*! \brief Compute the distance between a monster and a given position */
int monster_distance(Level *level, monster *m, int row, int col);

/*! \brief Check whether a monster has line of sight to a given position */
int monster_LoS_to(Level *level, monster *m, int row, int col);

#endif /* ENEMY_ACTIONS_H */
