#include "level.h"
#include <string.h>
#include "utils.h"
#include "printer.h"

Level *load_level_map(char *filename){
	int i, nchars;
	FILE *fin=open_file(filename,"r");
	Level *level=dmalloc(sizeof(Level));

	/* Read n cols and n rows from file */
	fscanf(fin,"%d\n%d\n", &(level->rows), &(level->cols));
	level->map=dmalloc(sizeof(char *)*level->rows);
	level->explored=dmalloc(sizeof(char *)*level->rows);
	
	for(i=0; i<level->rows; i++){
		level->map[i]=dmalloc(sizeof(char *)*level->cols);
		level->explored[i]=dmalloc(sizeof(char *)*level->cols); 
		/*! Note: explored is automatically initialized to 0 by dmalloc */
		nchars=fscanf(fin,"%s\n", level->map[i]);
		if(nchars<1 || strlen(level->map[i])<level->cols) {
			fprintf(stderr,"Read error at line %d (rows: %d, cols: %d): %s\n", i, 
							level->rows, level->cols, level->map[i]);
			fprintf(stderr,"Items read: %d; string length: %d\n", nchars, 
							(int)strlen(level->map[i]));
			return NULL;		
		}
		
	}
	return level;
}

int checkname(Level *level, char *filename){
	return strcmp(level->filename,filename);
}

int cmp_level_depth(Level *l1, Level *l2){
	return l1->depth-l2->depth;
}

elem *load_level(elem *levels, char *filename){
	/*! \todo Add checks to avoid that items be placed outside the map bounds 
	 *  This is a significant issue which can cause segfault when applied, e.g.,
	 *  to stairs, which modify the basic map.
	 */
	int i, depth, end=1;
	char buf1[128];
	char buf2[128];
	Level *level=NULL;
	FILE *fin=open_file(filename, "r");

	fscanf(fin,"name:%s\n",buf1);
	fscanf(fin,"depth:%d\n",&depth);
	fscanf(fin,"map:%s\n", buf2);
	level=load_level_map(buf2);
	insert(levels,level);

	level->name=strdup(buf1);
	level->filename=basename(strdup(filename));
	level->depth=depth;
	level->objs=dmalloc(sizeof(content)*MAXITEMS);
	reset(level->objs,sizeof(content)*MAXITEMS);
	
	for(i=0; end!=EOF && i<MAXITEMS; i++){
		int x, y, n, m, s;
		char sym;
		unsigned itype;
		end=fscanf(fin,"[%d,%d]:%c ", &x, &y, &sym);
		if (end<3) continue;
		if (feof(fin)) break;
		debug("reading symbol %c in file %s", sym, level->filename);
		itype=get_item_type_from_symbol(sym);
		if (itype!=NOITEM) {
			end=fscanf(fin,"%d %d %d\n", &n, &m, &s);
			add_item_to_level(level, x, y, itype, n, m, NULL, s);
		} else {
			level->objs[i].stash=NULL;
		}
		itype=get_monster_type_from_symbol(sym);
		if (itype!=NOITEM) {
			end=fscanf(fin,"\n");
			add_monster_to_level(level, x, y, itype);
		} else {
			level->objs[i].npc=NULL;
		}
		level->objs[i].special=0;
		/* Handle stairs */
		if (sym=='>' || sym=='<') {
			char buf3[128];
			elem *next;
			end=fscanf(fin,"%s\n", buf3);
			if (strcmp(buf3,"exit")==0) {
				/* A 0x1 next level means the stairs lead to 
				 * an exit from the dungeon */
				add_stairs_to_level(level,x,y,sym,(Level *)0x1);
			} else {
				next = search_list(levels,buf3,(int (*)(void *, void*)) checkname);
				if (!next) levels=load_level(levels, buf3);
				next = search_list(levels,buf3,(int (*)(void *, void*)) checkname);
				add_stairs_to_level(level,x,y,sym,next->data);
			}
		}
	}
	level->objs[i].valid=0;
	
	return levels;
}

void print_level(Level *level, int row, int col, char pcsymbol){
	char **map;
	int i, j;
	/*! Create memory for the map and copy the base map from level */
	map=dmalloc(sizeof(char *)*level->rows);
	for(i=0; i<level->rows; i++){
		map[i]=strdup(level->map[i]);
	}
	/*! Add symbols for objects and monsters 
	 * \todo Monsters should be added after items, actually.
	 */
	for(i=0; level->objs[i].valid!=0 && i<MAXITEMS; i++){
		int x, y;
		x=level->objs[i].row;
		y=level->objs[i].col;
		if (level->objs[i].valid==VALID && line_of_sight(level,row,col,x,y)) {
			if (level->objs[i].stash) map[x][y]=get_item_symbol(level->objs[i].stash);
			if (level->objs[i].npc) map[x][y]=level->objs[i].npc->species->symbol;
		}
	}
	/*! Add the symbol corresponding to the PC */
	map[row][col]=pcsymbol;
	
	/*! Blank out unexplored regions */
	for(i=0; i<level->rows; i++)
		for(j=0; j<level->cols; j++){
			if (!level->explored[i][j]) map[i][j]=' ';
		}
	
	/*! Print out the map using the printer.h facilities */
	print_map(map, level->rows);
	
	/*! Deallocate the memory used for the map */
	for(i=0; i<level->rows; i++){
		dfree(map[i]);
	}
	dfree(map);
}


void add_stairs_to_level(Level *level, int row, int col, char sym, Level *dest){
	int i = count_objs_in_level(level);
	level->objs[i].row=row;
	level->objs[i].col=col;
	level->objs[i].stairs_to=dest;
	level->objs[i].valid=VALID;
	level->map[row][col]=sym;
}


int count_objs_in_level(Level *level){
	int i;
	for(i=0; level->objs[i].valid!=INVALID && i<MAXITEMS; i++);
	return i;
}

void add_item_to_level(Level *level, int row, int col, item_types itype,
											 int quantity, int magic, char *text, int special){
		int i = count_objs_in_level(level);
		level->objs[i].row=row;
		level->objs[i].col=col;
		level->objs[i].stash=create_item(itype, quantity, magic, text, special);
		level->objs[i].valid=VALID;
}

void *remove_item_from_level(Level *level, int i) {
	void *res=NULL;
	level->objs[i].valid=REMOVED;
	if (level->objs[i].stash) res=level->objs[i].stash;
	if (level->objs[i].npc) res=level->objs[i].npc;
	level->objs[i].npc=NULL;
	level->objs[i].stash=NULL;
	return res;
}

void add_monster_to_level(Level *level, int row, int col, monsters type){
	int i = count_objs_in_level(level);
	level->objs[i].row=row;
	level->objs[i].col=col;
	level->objs[i].npc=create_monster(type);
	level->objs[i].valid=VALID;
}

void remove_monster_from_level(Level *level, monster *npc){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].npc == npc) {
			destroy_monster(level->objs[i].npc);
			level->objs[i].valid=REMOVED;
		}		 
}

monster *find_monster_at(Level *level, int row, int col){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].valid==VALID && level->objs[i].row==row && 
		    level->objs[i].col==col && level->objs[i].npc!=NULL)
			return level->objs[i].npc;
	return NULL;
}

Level *get_level_entered_at(Level *level, int row, int col){
	int i;
	for(i=0; i<count_objs_in_level(level); i++)
		if (level->objs[i].valid==VALID && level->objs[i].row==row && 
		    level->objs[i].col==col)
			return level->objs[i].stairs_to;
	return NULL; 
}

occupation can_move_to(Level *level, int row, int col){
	if (level->map[row][col]=='<') return STAIRS_UP;
	if (level->map[row][col]=='>') return STAIRS_DOWN;
	if (level->map[row][col]!='.') return BUSY;
	if (find_monster_at(level, row, col)) return ATCK;
	return FREE;
}


int block_los(Level *level, int row, int col){
	switch (level->map[row][col]) {
		case '#' : /* Add all other opaque blocks here */
		case '8' : return TRUE;
		default : return FALSE;
	}
}


int line_of_sight(Level *level, int a_row, int a_col, int b_row, int b_col){
	/*! Implementation based on Wikipedia's version of Bresenham's line algorithm 
	 *  http://en.wikipedia.org/wiki/Bresenham's_line_algorithm
	 */
	int dx, dy, sx, sy, x0, x1, y0, y1, err, e2;
	x0=a_row;
	x1=b_row;
	y0=a_col;
	y1=b_col;

	/* Using the three operand expression of C, which evaluates to the second
	 * operand is the first is true, or to the third operand otherwise  */
	dx = (x1>x0 ? x1-x0 : x0-x1);
	dy = (y1>y0 ? y1-y0 : y0-y1);
	sx = (x0<x1 ? 1 : -1);
	sy = (y0<y1 ? 1 : -1);
	err = dx-dy;

  if (x0==x1 && y0==y1) return TRUE;
	while(1){ 
     e2=2*err;
     if (e2>-dy) {
       err-=dy;
       x0+=sx;
     }
     if (x0==x1 && y0==y1) return TRUE;
     if (e2 <  dx) {
       err+=dx;
       y0+=sy;
     }
     if (x0==x1 && y0==y1) return TRUE;
     if (block_los(level, x0, y0)) return FALSE;
	}

	printf("We've got a problem here...\n");
	return TRUE;
}

void mark_explored(Level *level, int row, int col){
	int i, j;
	/*! Blank out unexplored regions */
	level->explored[row][col]=TRUE;
	for(i=0; i<level->rows; i++)
		for(j=0; j<level->cols; j++){
			if (line_of_sight(level, row, col, i, j)) level->explored[i][j]=TRUE;
		}
}


#ifdef TEST_LEVEL
void print_end(char *s) {
	#ifdef CURSES
		refresh();
		napms(1000);
		mvprintw(LINES-1, COLS-11,s); 
		refresh();
		napms(1000);
	#endif /* CURSES */
}

int main(int argc, char **argv){
	elem *levels;
	Level *level;
	
	screen_init();
	levels=load_level(NULL,argv[1]);
	level=levels->data;
	mark_explored(level,8,2);
	print_level(level,8,2,'@');
	print_end("Curses Test");
	add_item_to_level(level, 7, 4, SCROLL, 1, 1, 0, 0);
	print_level(level,8,2,'@');
	print_end("%% Added    "); 

	while(levels!=NULL) {
	level=levels->data;
	print_msg("Level %s", level->name);
	mark_explored(level,8,2);
	print_level(level,8,2,'@');
	print_end("Scanning levels");
	levels=levels->next;
	}
	screen_close();
	
	#ifdef CURSES
		printf("Test finished with curses\n");
	#else
		printf("Test finished without curses\n");
	#endif /* CURSES */
}
#endif /* TEST_LEVEL */
