#ifndef CHARACTER_H
#define CHARACTER_H
/*! \file character.h 
 * \brief Definition of the Player Character and related functions (generation, printing) 
 */
#include "items.h"

/*! \brief Enumerator type for character status, from hale to dead */
typedef enum _character_status {
	HALE /*! No penalties */,      
	WEAKENED /*! -3 to all actions */,  
	IMPAIRED /*! -6 to all actions */,  
	DEAD /*! Game over... */      
} character_status;

/*! \brief Character class definition; used for player characters */
typedef struct _class {
	char name[16]; /*! Name of the character class */
	int dmg_melee; /*! Damage bonus in melee attacks */
	int dmg_ranged;/*! Damage bonus in ranged attacks */
	int dmg_spell; /*! Damage bonus in spell attacks */
	int att_melee; /*! To hit bonus in melee attacks */
	int att_ranged;/*! To hit bonus in ranged attacks */
	int att_spell; /*! To hit bonus in spell attacks */
	int def_might; /*! Defense bonus against poison attacks */
	int def_speed; /*! Defense bonus against melee/ranged/generic spell attacks */
	int def_mind;  /*! Defense bonus against mind attacks */
} char_class;

/*! \brief Indices of classes in the \see char_class data structure */
typedef enum _classes {
	FIGHTER, THIEF, WIZARD, NCLASSES /*! NCLASSES serves as the counter of character classes  */
} classes;

/*! \brief Player Character type
 * 
 *  Type definition for the player character
 *  The character has three main stats
 *  These may be damaged by attacks and provide bonuses to most rolls
 */ 
typedef struct _player {
	/* Generalities */
	char *name;    /*! ask player */
	char symbol;   /*! def = @ */
	/* Statistics */
	int might;     /*! 2d6+6 */
	int speed;     /*! 2d6+6 */
	int mind;      /*! 2d6+6 */
	/*! Current values */
	int cur_might; /*! max = might */
	int cur_speed; /*! max = speed */
	int cur_mind;  /*! max = mind  */
	/* Progression */
	char_class *class; 
	int level;   /*! 1+ */
	int xp;      /*! every 16 xp -> +1 level */
	/* Position */
	int dungeon; /*! 1+ */
	int row;     /*! 1-18 */
	int col;     /*! 1-38 */
	unsigned turn;    /*! init at 0 */
	/* Equipment */
	item *equipment[25]; /*! max 25 object types (a-z) */
	item *wielded;
	item *quiver;
} character;


/*! \brief Random generation of player characters
 *
 * \todo Exercise: implement a non-random generator with character interaction
 * \return A character
 */
character *random_create_player_character();

/*! \brief Compute the bonus to might actions

 *   \return (might - 8) /3
 */
int character_might_bonus(character *pc);

/*! \brief Compute the bonus to speed actions
 *
 *   \return (speed - 8) /3
 */
int character_speed_bonus(character *pc);

/*! \brief Compute the bonus to mind actions
 *
 *   \return (mind - 8) /3
 */
int character_mind_bonus(character *pc);

/*! \brief Compute the status of the PC from his/her stats */
character_status character_get_status(character *pc);

/*! \brief Compute to hit bonus for melee attacks */
int character_melee_attack_bonus(character *pc);

/*! \brief Compute damage bonus for melee attacks */
int character_melee_damage_bonus(character *pc);


/*! \brief Initialize the memory used for character representation */
char **init_text_compact();

/*! \brief Finalize the memory used for character representation */
void destroy_text_compact(char **res);

/*! \brief Store a representation of the characters in a small space */
char **character_to_text_compact(char **res, character *pc);


#endif /* CHARACTER_H */
