#include "actions.h"
#include "window.h"

/*! \mainpage 
 * 
 * This is a Rogue-like game, mostly inspired by Dungeon Crawl.
 * It currently supports only exploration of a single level, with many
 * features to be implemented as exercises.
 * 
 *
 *  \section Design
 *  Like any game, this rogue-like is composed of a main loop that runs a set of
 *  fixed steps: refreshing the screen with updated information (map, character
 *  status); accepting an action command from the player; executing the command
 *  and computing and applying any effect; increasing the turn counter.
 *
 *  Note that several elements of the game such as monsters and items appear 
 *  several time in a level/game. Each instance has some specific properties,
 *  but all instances share a lot of common properties (e.g., all scrolls are
 *  represented in the game with the '%' symbol.
 *  Thus, items and monsters are represented as structures that include a pointer
 *  to a "class" or "type" structure. There is a single "scroll" structure, which
 *  defines properties common to all scrolls, and which is pointed to by all 
 *  scroll structures.
 *  This design is very general, and can be applied in many cases. 
 *
 *  \section Testing
 *  Note that several files support testing of the specific library
 *  This is obtained through a main function enclosed within "ifdef" macros.
 *  A specific "define" is used for each file, name TES_<filename> so as
 *  to allow selective activation of these test programs.
 *  Running make will compile the code and run the tests for all files 
 *  that have been modified.
 *
 *  \section extension Extending and Improving the Game
 *  As mentioned, there are many missing features that may be implemented as
 *  an exercise in programming. The following are the main missing features:
 * 
 *  \todo Implement inventory (pick/drop items)
 *  \todo Implement monster movement (random)
 *  \todo Implement melee combat (monster attacks and turns)
 *  \todo Implement usable items (eat/use items)
 *  \todo Implement save and load game
 *  \todo Implement support for multiple (connected) levels
 *  \todo Implement line of sight
 *  \todo Implement monster movement (follow PC)
 *  \todo Implement ranged combat
 *  \todo Implement spellcasting
 *  \todo Implement discovery of items
 *  \todo Implement item subclasses (e.g., scroll of summoning, scroll of enchanting, etc.)
 *  \todo Implement discovery of areas
 *  \todo Implement fog of war
 *  \todo Implement arbitrary sized maps with scrolling
 *  \todo Implement different speeds for monsters and PCs
 *  \todo Implement more items (armor, more weapons, rods/wands, rings, amulets, clothing)
 *  \todo Implement more special effects (traps, illusory walls, teleports)
 *  \todo Implement special status conditions for PCs and monsters (e.g., invisible, poisoned, confused, slowed, etc.)
 *
 *  See also the TODO list for more ideas.
 *
 *  \see main for an overview of the main game loop
 *  \see window.h for configuration information
 *  \see actions.h for the execution of each game action 
 *  \see character.h for the definition of the player character
 *  \see commands.h for the definition and acquisition of a command
 *  \see items.h for the definition of items found in the game
 *  \see level.h for the definition of the game Level
 *  \see printer.h for the printing and screen management facilities
 *  \see utils.h for assorted utilities
 */

/*
 *  \brief Rogue-like game, main function
 * 
 *  \param argc number of arguments: should be 2
 *  \param argv contains the level to be loaded
 *
 *  The main loop of the game performs printing and 
 *  interprets the actions.
 */
int main(int argc, char **argv){
	/*!  Initialization (random character only, for now) */
	command cmd=NOCMD;
	char **desc=init_text_compact();
	Level *level=load_level(argv[1]);
	character *pc;
	screen_init(); 
	init_random_generator(); 
	pc=pc=random_create_player_character();
	
	/*!  Loop until the user gives a quit/save command */
	while(cmd!=SAVE) {
		/*!    - Print routine */
		print_level(level,pc->row,pc->col,pc->symbol);
		character_to_text_compact(desc,pc);
		print_strings(desc,COMPACT_TEXT_ROWS,STATUS_COL);

		/*!    - Get and execute action, and advance time by 1 turn */
		pc->turn+=1;
		cmd=get_command();
		action(level,pc,cmd);
	}
	
	/*!  Finalization */
	destroy_text_compact(desc);
	screen_close();
}
