#ifndef UTILS_H
#define UTILS_H
/*! \file utils.h
 *  \brief Various utilities, primarily in dynamic memory management 
 *         random generation, and debugging
 */
#include <stdio.h>

/*! \name File management */
/**@{*/
/*! \brief Open file for reading with error management */
FILE *open_file(char *filename, char *mode);
/**@}*/

/*! \name Dynamic memory allocation */
/**@{*/
/*! \brief Allocate size bytes of memory, with error management */
void *dmalloc(size_t size);

/*! \brief Free data memory */
void dfree(void *data);

/*! \brief Zero size bytes of data */
void reset(void *data, size_t size);
/**@}*/

/*! \name random Random generator */
/**@{*/
/*! \brief Initialize random number generator */
void init_random_generator();

/*! \brief Roll a d20 */
int d20();

/*! \brief Roll n d6 
 *
 * \param n  The number of d6 to roll
 * \return The sum of n d6
 */
int d6(int n);

/*! \brief Get a generic random int between bt and tp included */
int randint(int bt, int tp);

/*! \brief Load a random name from the random names file */
char *random_name();
/**@}*/

/*! \brief Debug support 
 * 
 * This function acts like the typical printf, except
 * it prints to stderr and prefixes the string to print
 * with the name of the file where the print happens and
 * the line number (useful to see what is going on with
 * the program)
 */
#ifdef DEBUG
	#define debug(S, ...) fprintf(stderr,"%s:%d " S "\n", __FILE__, __LINE__, ##__VA_ARGS__)
#else 
	#define debug(S, ...)
#endif /* DEBUG */

#endif /* UTILS_H */
