/*
 * Andrea Di Biagio
 * Politecnico di Milano, 2007
 * 
 * axe_debug.c
 * Formal Languages & Compilers Machine, 2007/2008
 * 
 */

#include "axe_debug.h"
#include "collections.h"
#include "reg_alloc_constants.h"
#include "cflow_constants.h"


static void printBBlockInfos(t_basic_block *block, FILE *fout, int verbose);
static void printLiveIntervals(t_list *intervals, FILE *fout);
static void printBindings(int *bindings, int numVars, FILE *fout);

void printBindings(int *bindings, int numVars, FILE *fout)
{
   int counter;
   
   if (bindings == NULL)
      return;

   if (fout == NULL)
      return;

   /* initialize counter */
   counter = 0;
   fprintf(fout, "BINDINGS : \n");
   while(counter <= numVars)
   {
      if (bindings[counter] != RA_SPILL_REQUIRED)
      {
         fprintf(fout, "VAR T%d will be assigned to register R%d \n"
                  , counter, bindings[counter]);
      }
      else
      {
         fprintf(fout, "VAR T%d will be spilled \n", counter);
      }
      
      counter++;
   }
}

void printRegAllocInfos(t_reg_allocator *RA, FILE *fout)
{
   if (RA == NULL)
      return;
   if (fout == NULL)
      return;
   fprintf(fout, "\n\n*************************\n");
   fprintf(fout, "REGISTER ALLOCATION INFOS\n");
   fprintf(fout, "*************************\n");
   fprintf(fout, "AVAILABLE REGISTERS : %d \n", RA->regNum + 3);
   fprintf(fout, "USED VARIABLES : %d \n", RA->varNum);
   fprintf(fout, "-------------------------\n");
   printLiveIntervals(RA->live_intervals, fout);
   fprintf(fout, "-------------------------\n");
   printBindings(RA->bindings, RA->varNum, fout);
   fprintf(fout, "*************************\n\n");
}

void printLiveIntervals(t_list *intervals, FILE *fout)
{
   t_list *current_element;
   t_live_interval *interval;

   /* precondition */
   if (fout == NULL)
      return;

   fprintf(fout, "LIVE_INTERVALS:\n");

   /* retireve the first element of the list */
   current_element = intervals;
   while (current_element != NULL)
   {
      interval = (t_live_interval *) LDATA(current_element);

      fprintf(fout, "\tLIVE_INTERVAL of T%d : [%d, %d] \n"
            , interval->varID, interval->startPoint, interval->endPoint);
      
      /* retrieve the next element in the list of intervals */
      current_element = LNEXT(current_element);
   }
}

void printBBlockInfos(t_basic_block *block, FILE *fout, int verbose)
{
   t_list *current_element;
   t_cflow_Node *current_node;
   int count;
   
   /* preconditions */
   if (block == NULL)
      return;
   if (fout == NULL)
      return;

   fprintf(fout,"NUMBER OF PREDECESSORS : %d \n"
         , getLength(block->pred) );
   fprintf(fout,"NUMBER OF SUCCESSORS : %d \n"
         , getLength(block->succ) );
   fprintf(fout,"NUMBER OF INSTRUCTIONS : %d \n"
         , getLength(block->nodes) );

   count = 1;
   current_element = block->nodes;
   while(current_element != NULL)
   {
      current_node = (t_cflow_Node *) LDATA(current_element);
      fprintf(fout,"\t%d.  ", count);
      debug_printInstruction(current_node->instr, fout);
      if (verbose != 0)
      {
         if (current_node->def != NULL)
            fprintf(fout, "\n\t\t\tDEF = [R%d]", (current_node->def)->ID);
         if (current_node->uses[0] != NULL)
         {
            fprintf(fout, "\n\t\t\tUSES = [R%d", ((current_node->uses)[0])->ID);
            if (current_node->uses[1] != NULL)
               fprintf(fout, ", R%d", ((current_node->uses)[1])->ID);
            if (current_node->uses[2] != NULL)
               fprintf(fout, ", R%d", ((current_node->uses)[2])->ID);
            fprintf(fout, "]");
         }
         fprintf(fout, "\n\t\t\tLIVE IN = [");
         printListOfVariables(current_node->in, fout);
         fprintf(fout, "]");
         fprintf(fout, "\n\t\t\tLIVE OUT = [");
         printListOfVariables(current_node->out, fout);
         fprintf(fout, "]");
      }
      
      fprintf(fout, "\n");
      count++;
      current_element = LNEXT(current_element);
   }
}

void printListOfVariables(t_list *variables, FILE *fout)
{
   t_list *current_element;
   t_cflow_var *current_variable;
   
   if (variables == NULL)
      return;
   if (fout == NULL)
      return;

   current_element = variables;
   while(current_element != NULL)
   {
      current_variable = (t_cflow_var *) LDATA(current_element);
      fprintf(fout, "R%d", current_variable->ID);
      if (LNEXT(current_element) != NULL)
         fprintf(fout, ", ");
      
      current_element = LNEXT(current_element);
   }
}

void printGraphInfos(t_cflow_Graph *graph, FILE *fout, int verbose)
{
   int counter;
   t_list *current_element;
   t_basic_block *current_bblock;
   
   /* preconditions */
   if (graph == NULL)
      return;
   if (fout == NULL)
      return;

   /* initialization of the local variables */
   counter = 1;
   
   fprintf(fout,"NOTE : Temporary registers are considered as\n"
                "       variables of the intermediate language. \n");
#ifdef CFLOW_ALWAYS_LIVEIN_R0
   fprintf(fout,"       Variable \'R0\' (that refers to the \n"
                "       physical register \'RO\') is always \n"
                "       considered LIVE-IN for each node of \n"
                "       a basic block. \n"
                "       Thus, in the following control flow graph, \n"
                "       \'R0\' will never appear as LIVE-IN or LIVE-OUT\n"
                "       variable for a statement.\n\n"
                "       If you want to consider \'R0\' as\n"
                "       a normal variable, you have to set\n"
                "       to 0 the value of the macro CFLOW_ALWAYS_LIVEIN_R0\n"
                "       defined in \"cflow_constants.h\".\n\n");
#endif
   fprintf(fout,"\n");
   fprintf(fout,"**************************\n");
   fprintf(fout,"     CONTROL FLOW GRAPH   \n");
   fprintf(fout,"**************************\n");
   fprintf(fout,"NUMBER OF BASIC BLOCKS : %d \n"
         , getLength(graph->blocks));
   fprintf(fout,"NUMBER OF USED VARIABLES : %d \n"
         , getLength(graph->cflow_variables));
   fprintf(fout,"--------------------------\n");
   fprintf(fout,"START BASIC BLOCK INFOS.  \n");
   fprintf(fout,"--------------------------\n");

   /* initialize `current_block' */
   current_element = graph->blocks;
   while(current_element != NULL)
   {
      current_bblock = (t_basic_block *) LDATA(current_element);
      fprintf(fout,"[BLOCK %d] \n", counter);
      printBBlockInfos(current_bblock, fout, verbose);
      if (LNEXT(current_element) != NULL)
         fprintf(fout,"--------------------------\n");
      else
         fprintf(fout,"**************************\n");

      counter++;
      current_element = LNEXT(current_element);
   }
   
   fprintf(fout,"\n\n");
}


void debug_printInstruction(t_axe_instruction *instr, FILE *fout)
{
   /* preconditions */
   if (fout == NULL)
      return;
   
   if (instr == NULL)
   {
      fprintf(fout, "[NULL] \n");
      return;
   }

   if (instr->labelID != NULL)
      fprintf(fout, "L%d\t", (instr->labelID)->labelID);
   else
      fprintf(fout, "\t");
   
   switch(instr->opcode)
   {
      case ADD : fprintf(fout, "add "); break;
	case ADDU : fprintf(fout, "addu "); break;
	case AND : fprintf(fout, "and "); break;
	case DIV : fprintf(fout, "div "); break;
	case DIVU : fprintf(fout, "divu "); break;
	case MUL : fprintf(fout, "mul "); break;
	case MULO : fprintf(fout, "mulo "); break;
	case MULOU : fprintf(fout, "mulou "); break;
	case NOR : fprintf(fout, "nor "); break;
	case OR : fprintf(fout, "or "); break;
	case REM : fprintf(fout, "rem "); break;
	case REMU : fprintf(fout, "remu "); break;
	case SLLV : fprintf(fout, "sllv "); break;
	case SRAV : fprintf(fout, "srav "); break;
	case SRLV : fprintf(fout, "srlv "); break;
	case ROL : fprintf(fout, "rol "); break;
	case ROR : fprintf(fout, "ror "); break;
	case SUB : fprintf(fout, "sub "); break;
	case SUBU : fprintf(fout, "subu "); break;
	case XOR : fprintf(fout, "xor "); break;
	case SLT : fprintf(fout, "slt "); break;
	case SLTU : fprintf(fout, "sltu "); break;
	case SEQ : fprintf(fout, "seq "); break;
	case SGE : fprintf(fout, "sge "); break;
	case SGEU : fprintf(fout, "sgeu "); break;
	case SGT : fprintf(fout, "sgt "); break;
	case SGTU : fprintf(fout, "sgtu "); break;
	case SLE : fprintf(fout, "sle "); break;
	case SLEU : fprintf(fout, "sleu "); break;
	case SNE : fprintf(fout, "sne "); break;
	case MOVN : fprintf(fout, "movn "); break;
	case MOVZ : fprintf(fout, "movz "); break;
	case ADDI : fprintf(fout, "addi "); break;
	case ADDIU : fprintf(fout, "addiu "); break;
	case ANDI : fprintf(fout, "andi "); break;
	case ORI : fprintf(fout, "ori "); break;
	case SLL : fprintf(fout, "sll "); break;
	case SRA : fprintf(fout, "sra "); break;
	case SRL : fprintf(fout, "srl "); break;
	case XORI : fprintf(fout, "xori "); break;
	case SLTI : fprintf(fout, "slti "); break;
	case SLTIU : fprintf(fout, "sltiu "); break;
	case ABS : fprintf(fout, "abs "); break;
	case CLO : fprintf(fout, "clo "); break;
	case CLZ : fprintf(fout, "clz "); break;
	case DV : fprintf(fout, "dv "); break;
	case DVU : fprintf(fout, "dvu "); break;
	case MULT : fprintf(fout, "mult "); break;
	case MULTU : fprintf(fout, "multu "); break;
	case MADD : fprintf(fout, "madd "); break;
	case MADDU : fprintf(fout, "maddu "); break;
	case MSUB : fprintf(fout, "msub "); break;
	case NEG : fprintf(fout, "neg "); break;
	case NEGU : fprintf(fout, "negu "); break;
	case NOTB : fprintf(fout, "not "); break;
	case JALR : fprintf(fout, "jalr "); break;
	case TEQ : fprintf(fout, "teq "); break;
	case TNE : fprintf(fout, "tne "); break;
	case TGE : fprintf(fout, "tge "); break;
	case TGEU : fprintf(fout, "tgeu "); break;
	case TLT : fprintf(fout, "tlt "); break;
	case TLTU : fprintf(fout, "tltu "); break;
	case MOVE : fprintf(fout, "move "); break;
	case LUI : fprintf(fout, "lui "); break;
	case LI	 : fprintf(fout, "li "); break;
	case TEQI : fprintf(fout, "teqi "); break;
	case TNEI : fprintf(fout, "tnei "); break;
	case TGEI : fprintf(fout, "tgei "); break;
	case TGEIU : fprintf(fout, "tgeiu "); break;
	case TLTI : fprintf(fout, "tlti "); break;
	case TLTIU : fprintf(fout, "tltiu "); break;
	case JR : fprintf(fout, "jr "); break;
	case MFHI : fprintf(fout, "mfhi "); break;
	case MFLO : fprintf(fout, "mflo "); break;
	case MTHI : fprintf(fout, "mthi "); break;
	case MTLO : fprintf(fout, "mtlo "); break;
	case LA : fprintf(fout, "la "); break;
	case LB : fprintf(fout, "lb "); break;
	case LBU : fprintf(fout, "lbu "); break;
	case LH : fprintf(fout, "lh "); break;
	case LHU : fprintf(fout, "lhu "); break;
	case LW : fprintf(fout, "lw "); break;
	case LWC1 : fprintf(fout, "lwc1 "); break;
	case LWL : fprintf(fout, "lwl "); break;
	case LWR : fprintf(fout, "lwr "); break;
	case LD : fprintf(fout, "ld "); break;
	case ULH : fprintf(fout, "ulh "); break;
	case ULHU : fprintf(fout, "ulhu "); break;
	case ULW : fprintf(fout, "ulw "); break;
	case SB : fprintf(fout, "sb "); break;
	case SH : fprintf(fout, "sh "); break;
	case SW : fprintf(fout, "sw "); break;
	case SWC1 : fprintf(fout, "swc1 "); break;
	case SDC1 : fprintf(fout, "sdc1 "); break;
	case SWL : fprintf(fout, "swl "); break;
	case SWR : fprintf(fout, "swr "); break;
	case SD : fprintf(fout, "sd "); break;
	case USH : fprintf(fout, "ush "); break;
	case USW : fprintf(fout, "usw "); break;
	case BEQ : fprintf(fout, "beq "); break;
	case BNE : fprintf(fout, "bne "); break;
	case BGE : fprintf(fout, "bge "); break;
	case BGEU : fprintf(fout, "bgeu "); break;
	case BGT : fprintf(fout, "bgt "); break;
	case BGTU : fprintf(fout, "bgtu "); break;
	case BLE : fprintf(fout, "ble "); break;
	case BLEU : fprintf(fout, "bleu "); break;
	case BLT : fprintf(fout, "blt "); break;
	case BLTU : fprintf(fout, "bltu "); break;
	case BNEZ : fprintf(fout, "bnez "); break;
	case BGEZ : fprintf(fout, "bgez "); break;
	case BGEZAL : fprintf(fout, "bgezal "); break;
	case BGTZ : fprintf(fout, "bgtz "); break;
	case BLEZ : fprintf(fout, "blez "); break;
	case BLTZAL : fprintf(fout, "bltzal "); break;
	case BLTZ : fprintf(fout, "bltz "); break;
	case BEQZ : fprintf(fout, "beqz "); break;
	case B : fprintf(fout, "b "); break;
	case J : fprintf(fout, "j "); break;
	case JAL : fprintf(fout, "jal "); break;
	case NOP : fprintf(fout, "nop "); break;
	case SYSCALL : fprintf(fout, "syscall "); break;
      case INVALID_OPCODE : fprintf(fout, "[INVALID] ");
   }

   if (instr->reg_1 != NULL)
   {
      if (!(instr->reg_1)->indirect)
         fprintf(fout, "R%d ", (instr->reg_1)->ID);
      else
         fprintf(fout, "(R%d) ", (instr->reg_1)->ID);
   }
   if (instr->reg_2 != NULL)
   {
      if (!(instr->reg_2)->indirect)
         fprintf(fout, "R%d ", (instr->reg_2)->ID);
      else
         fprintf(fout, "(R%d) ", (instr->reg_2)->ID);
      if (instr->reg_3 != NULL)
      {
         if (!(instr->reg_3)->indirect)
            fprintf(fout, "R%d ", (instr->reg_3)->ID);
         else
            fprintf(fout, "(R%d) ", (instr->reg_3)->ID);
      }
      else
         fprintf(fout, "#%d ", instr->immediate);
   }
   
   if (instr->address != NULL)
   {
      if ((instr->address)->type == LABEL_TYPE)
         fprintf(fout, "L%d ", ((instr->address)->labelID)->labelID);
      else
         fprintf(fout, "%d ", (instr->address)->addr);
   }
}

char * dataTypeToString(int codedType)
{
   switch (codedType)
   {
      case INTEGER_TYPE : return "INTEGER";
      default : return "<INVALID_TYPE>";
   }
}
