/*
 * Andrea Di Biagio
 * Politecnico di Milano, 2007
 * 
 * axe_engine.c
 * Formal Languages & Compilers Machine, 2007/2008
 * 
 */

#include <assert.h>
#include <string.h>
#include <malloc.h>
#include "axe_engine.h"
#include "symbol_table.h"
#include "axe_errors.h"

/* global variable errorcode */
int errorcode;

static int getInstructionGroup(int opcode);
/* Function used when a compare is needed between two labels */
static int compareVariables (void *Var_A, void *Var_B);

/* Finalize the memory associated with an instruction */
static void finalizeInstructions(t_list *instructions);

/* Finalize the data segment */
static void finalizeDataSegment(t_list *dataDirectives);

/* finalize the informations associated with all the variables */
static void finalizeVariables(t_list *variables);

/* Translate the assembler directives (definitions inside the data segment */
static void translateDataSegment(t_program_infos *program, FILE *fp);

/* Translate all the instructions within the code segment */
static void translateCodeSegment(t_program_infos *program, FILE *fp);

/* print out to the file `fp' an opcode */
static void printOpcode(int opcode, FILE *fp);

/* print out to the file `fp' a register information */
static void printRegister(t_axe_register *reg, FILE *fp);

/* add a variable to the program */
static void addVariable(t_program_infos *program, t_axe_variable *variable);

      
/* create a new variable */
void createVariable(t_program_infos *program, char *ID
      , int type, int isArray, int arraySize, int init_val)
{
   t_axe_variable *var;
         
   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* initialize a new variable */
   var = alloc_variable(ID, type, isArray, arraySize, init_val);
   if (var == NULL)
      notifyError(AXE_OUT_OF_MEMORY);
   
   /* assign a new label to the newly created variable `var' */
   var->labelID = reserveLabel(program);

   /* add the new variable to program */
   addVariable(program, var);
}

/* translate each instruction in his assembler symbolic representation */
void translateCodeSegment(t_program_infos *program, FILE *fp)
{
   t_list *current_element;
   t_axe_instruction *current_instruction;
   int _error;
   
   /* preconditions */
   if (fp == NULL)
      notifyError(AXE_INVALID_INPUT_FILE);

   if (program == NULL)
   {
      _error = fclose(fp);
      if (_error == EOF)
         notifyError(AXE_FCLOSE_ERROR);
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   }

   /* initialize the current_element */
   current_element = program->instructions;

   /* write the .text directive */
   if (current_element != NULL)
   {
      if (fprintf(fp, "\t.text\n") < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }
   }

   while (current_element != NULL)
   {
      /* retrieve the current instruction */
      current_instruction = (t_axe_instruction *) LDATA(current_element);
      assert(current_instruction != NULL);
      assert(current_instruction->opcode != INVALID_OPCODE);

      if (current_instruction->labelID != NULL)
      {
            /* create a string identifier for the label */
            if (  fprintf(fp, "L%d : \t"
                     , (current_instruction->labelID)->labelID ) < 0)
            {
              _error = fclose(fp);
               if (_error == EOF)
                  notifyError(AXE_FCLOSE_ERROR);
               notifyError(AXE_FWRITE_ERROR);
            }
      }
      else
      {
            /* create a string identifier for the label */
            if (fprintf(fp, "\t") < 0)
            {
              _error = fclose(fp);
               if (_error == EOF)
                  notifyError(AXE_FCLOSE_ERROR);
               notifyError(AXE_FWRITE_ERROR);
            }
      }

      /* print the opcode */
      printOpcode(current_instruction->opcode, fp);

      if (  getInstructionGroup(current_instruction->opcode) == SPECIAL )
      {
         if (fprintf(fp, "\n") < 0)
         {
              _error = fclose(fp);
               if (_error == EOF)
                  notifyError(AXE_FCLOSE_ERROR);
               notifyError(AXE_FWRITE_ERROR);
         }

         /* update the current_element */
         current_element = LNEXT(current_element);
         continue;
      }
      
      if (fputc(' ', fp) == EOF)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }

      if (current_instruction->reg_1 != NULL)
      {
         printRegister(current_instruction->reg_1, fp);
         
	 if(getInstructionGroup(current_instruction->opcode)==UNA){
		 current_element = LNEXT(current_element);
		 if (fprintf(fp, "\n") < 0)
		{
			_error = fclose(fp);
			if (_error == EOF)
			notifyError(AXE_FCLOSE_ERROR);
			notifyError(AXE_FWRITE_ERROR);
		}
		 continue;
	 }
	 
         if (fprintf(fp, ", ") < 0)
         {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }
	 
      }
      if((getInstructionGroup(current_instruction->opcode)==LOAD_STORE)&&
	  (current_instruction->addr_mode!=AM_INDIRECT))
      {
	      fprintf(stdout,"\nADDRESS MODE %d \n",current_instruction->addr_mode);
	      if (current_instruction->address != NULL)
		{
			if ((current_instruction->address)->type == ADDRESS_TYPE)
			{
				if (fprintf(fp, "%d", (current_instruction->address)->addr) < 0)
				{
					_error = fclose(fp);
					if (_error == EOF)
						notifyError(AXE_FCLOSE_ERROR);
					notifyError(AXE_FWRITE_ERROR);
				}
			}
			else
			{
				fprintf(stdout,"\nEL LABEL ES %d \n",((current_instruction->address)->labelID)->labelID);
				
				assert((current_instruction->address)->type == LABEL_TYPE);
				if (  fprintf(fp, "L%d", ((current_instruction->address)->labelID)->labelID) < 0)
				{
					_error = fclose(fp);
					if (_error == EOF)
						notifyError(AXE_FCLOSE_ERROR);
					notifyError(AXE_FWRITE_ERROR);
				}
			}
		}
	      switch(current_instruction->addr_mode)
	      {
		case AM_LABEL:
			current_element = LNEXT(current_element);
			if (fprintf(fp, "\n") < 0)
			{
				_error = fclose(fp);
				if (_error == EOF)
				notifyError(AXE_FCLOSE_ERROR);
				notifyError(AXE_FWRITE_ERROR);
			}
			continue;
			break;
		case AM_INDIRECT_LABEL_ADDRESS :
			fprintf(fp, "+");
			break;
		case AM_LABEL_IMMEDIATE :
			fprintf(fp, "+");
			current_element = LNEXT(current_element);
			if (fprintf(fp, "%d", current_instruction->immediate) < 0)
			{
				_error = fclose(fp);
				if (_error == EOF)
				notifyError(AXE_FCLOSE_ERROR);
				notifyError(AXE_FWRITE_ERROR);
			}
			if (fprintf(fp, "\n") < 0)
			{
				_error = fclose(fp);
				if (_error == EOF)
					notifyError(AXE_FCLOSE_ERROR);
				notifyError(AXE_FWRITE_ERROR);
			}
			continue;
			break;
	      }
      }
      
      if (current_instruction->reg_2 != NULL)
      {
         printRegister(current_instruction->reg_2, fp);
         if (errorcode != AXE_OK)
            return;

	if(getInstructionGroup(current_instruction->opcode)==BIN){
		 current_element = LNEXT(current_element);
		 if (fprintf(fp, "\n") < 0)
		{
			_error = fclose(fp);
			if (_error == EOF)
			notifyError(AXE_FCLOSE_ERROR);
			notifyError(AXE_FWRITE_ERROR);
		}
		 continue;
	 }
	 
	 if(getInstructionGroup(current_instruction->opcode)==LOAD_STORE)
	 {
		 current_element = LNEXT(current_element);
		 if (fprintf(fp, "\n") < 0)
		{
			_error = fclose(fp);
			if (_error == EOF)
			notifyError(AXE_FCLOSE_ERROR);
			notifyError(AXE_FWRITE_ERROR);
		}
		 continue;
	 }
	 
         if (fprintf(fp, ", ") < 0)
         {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }

      }
      if (current_instruction->reg_3 != NULL)
      {
         printRegister(current_instruction->reg_3, fp);

         if (fprintf(fp, "\n") < 0) {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }

         /* update the current_element */
         current_element = LNEXT(current_element);
         continue;
      }

      if (current_instruction->address != NULL)
      {
         if ((current_instruction->address)->type == ADDRESS_TYPE)
         {
            if (fprintf(fp, "%d", (current_instruction->address)->addr) < 0)
            {
               _error = fclose(fp);
               if (_error == EOF)
                  notifyError(AXE_FCLOSE_ERROR);
               notifyError(AXE_FWRITE_ERROR);
            }
         }
         else
         {
            assert((current_instruction->address)->type == LABEL_TYPE);
            if (  fprintf(fp, "L%d"
                     , ((current_instruction->address)->labelID)
                              ->labelID) < 0)
            {
               _error = fclose(fp);
               if (_error == EOF)
                  notifyError(AXE_FCLOSE_ERROR);
               notifyError(AXE_FWRITE_ERROR);
            }
         }
         
         if (fprintf(fp, "\n") < 0) {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }

         /* update the current_element */
         current_element = LNEXT(current_element);
         continue;
      }

      if (fprintf(fp, "%d", current_instruction->immediate) < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }

      if (fprintf(fp, "\n") < 0) {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }

      /* loop termination condition */
      current_element = LNEXT(current_element);
   }
}

void translateDataSegment(t_program_infos *program, FILE *fp)
{
   t_list *current_element;
   t_axe_data *current_data;
   int _error;
   int fprintf_error;
   
   /* preconditions */
   if (fp == NULL)
      notifyError(AXE_INVALID_INPUT_FILE);

   /* initialize the local variable `fprintf_error' */
   fprintf_error = 0;
   
   if (program == NULL)
   {
      _error = fclose(fp);
      if (_error == EOF)
         notifyError(AXE_FCLOSE_ERROR);

      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   }

   /* initialize the value of `current_element' */
   current_element = program->data;

   /* write the .data directive */
   if (current_element != NULL)
   {
      if (fprintf(fp, "\t.data\n") < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }
   }

   /* iterate all the elements inside the data segment */
   while (current_element != NULL)
   {
      /* retrieve the current data element */
      current_data = (t_axe_data *) LDATA(current_element);

      /* assertions */
      assert (current_data->directiveType != DIR_INVALID);

      /* create a string identifier for the label */
      if ( (current_data->labelID != NULL)
            && ((current_data->labelID)->labelID != LABEL_UNSPECIFIED) )
      {
         fprintf_error = fprintf(fp, "L%d : \t"
                  , (current_data->labelID)->labelID);
      }
      else
      {
         fprintf_error = fprintf(fp, "\t");
      }

      /* test if an error occurred while executing the `fprintf' function */
      if (fprintf_error < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }

      /* print the directive identifier */
      if (current_data->directiveType == DIR_WORD)
      {
         if (fprintf(fp, ".WORD ") < 0)
         {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }
      }
      
      else if (current_data->directiveType == DIR_SPACE)
      {
         if (fprintf(fp, ".SPACE ") < 0)
         {
            _error = fclose(fp);
            if (_error == EOF)
               notifyError(AXE_FCLOSE_ERROR);
            notifyError(AXE_FWRITE_ERROR);
         }
      }

      /* print the value associated with the directive */
      if (fprintf(fp, "%d\n", current_data->value) < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }

      /* loop termination condition */
      current_element = LNEXT(current_element);
   }
}

void finalizeDataSegment(t_list *dataDirectives)
{
   t_list *current_element;
   t_axe_data *current_data;

   /* nothing to finalize */
   if (dataDirectives == NULL)
      return;

   current_element = dataDirectives;
   while(current_element != NULL)
   {
      /* retrieve the current instruction */
      current_data = (t_axe_data *) LDATA(current_element);
      if (current_data != NULL)
         free_Data(current_data);

      current_element = LNEXT(current_element);
   }

   /* free the list of instructions */
   freeList(dataDirectives);
}

void finalizeInstructions(t_list *instructions)
{
   t_list *current_element;
   t_axe_instruction *current_instr;

   /* nothing to finalize */
   if (instructions == NULL)
      return;

   current_element = instructions;
   while(current_element != NULL)
   {
      /* retrieve the current instruction */
      current_instr = (t_axe_instruction *) LDATA(current_element);
      if (current_instr != NULL)
         free_Instruction(current_instr);

      current_element = LNEXT(current_element);
   }

   /* free the list of instructions */
   freeList(instructions);
}

int compareVariables (void *Var_A, void *Var_B)
{
   t_axe_variable *va;
   t_axe_variable *vb;
   
   if (Var_A == NULL)
   {
      if (Var_B == NULL)
         return 1;
   }

   if (Var_B == NULL)
      return 0;

   va = (t_axe_variable *) Var_A;
   vb = (t_axe_variable *) Var_B;

   /* test if the name is the same */
   return (!strcmp(va->ID, vb->ID));
}

t_axe_variable * getVariable
      (t_program_infos *program, char *ID)
{
   t_axe_variable search_pattern;
   t_list *elementFound;
   
   /* preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   if (ID == NULL)
      notifyError(AXE_VARIABLE_ID_UNSPECIFIED);

   /* initialize the pattern */
   search_pattern.ID = ID;
   
   /* search inside the list of variables */
   elementFound = CustomfindElement
         (program->variables, &search_pattern, compareVariables);

   /* if the element is found return it to the caller. Otherwise return NULL. */
   if (elementFound != NULL)
      return (t_axe_variable *) LDATA(elementFound);

   return NULL;
}

/* initialize an instance of `t_program_infos' */
t_program_infos * allocProgramInfos()
{
   t_program_infos *result;

   /* initialize the local variable `result' */
   result = (t_program_infos *)
         _AXE_ALLOC_FUNCTION(sizeof(t_program_infos));

   /* verify if an error occurred during the memory allocation
    * process */
   if (result == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* initialize the new instance of `result' */
   result->variables = NULL;
   result->instructions = NULL;
   result->data = NULL;
   result->current_register = 0; /* we are excluding the register R0 */
   result->lmanager = initialize_label_manager();

   if (result->lmanager == NULL)
   {
      finalizeProgramInfos(result);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   result->sy_table = initialize_sy_table();
   
   /* test if the sy_table is a NULL pointer */
   if (result->sy_table == NULL)
   {
      finalizeProgramInfos(result);
      notifyError(AXE_OUT_OF_MEMORY);
   }
   
   /* postcondition: return an instance of `t_program_infos' */
   return result;
}

/* add an instruction at the tail of the list `program->instructions'.
 * Returns an error code. */
void addInstruction(t_program_infos *program, t_axe_instruction *instr)
{
   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   
   if (instr == NULL)
      notifyError(AXE_INVALID_INSTRUCTION);

   if (program->lmanager == NULL)
      notifyError(AXE_INVALID_LABEL_MANAGER);

   instr->labelID = assign_label(program->lmanager);

   /* update the list of instructions */
   program->instructions = addElement(program->instructions, instr, -1);
}

/* reserve a new label identifier for future uses */
t_axe_label * reserveLabel(t_program_infos *program)
{
   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   if (program->lmanager == NULL)
      notifyError(AXE_INVALID_LABEL_MANAGER);

   return reserveLabelID(program->lmanager);
}

/* assign a new label identifier to the next instruction */
t_axe_label * fixLabel(t_program_infos *program, t_axe_label *label)
{
   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   if (program->lmanager == NULL)
      notifyError(AXE_INVALID_LABEL_MANAGER);

   /* fix the label */
   return fixLabelID(program->lmanager, label);
}

/* reserve a new label identifier */
t_axe_label * assignNewLabel(t_program_infos *program)
{
   t_axe_label * reserved_label;

   /* reserve a new label */
   reserved_label = reserveLabel(program);
   if (reserved_label == NULL)
      return NULL;

   /* fix the label */
   return fixLabel(program, reserved_label);
}

void addVariable(t_program_infos *program, t_axe_variable *variable)
{
   t_axe_variable *variableFound;
   t_axe_data *new_data_info;
   int sy_error;
   
   /* test the preconditions */
   if (variable == NULL)
      notifyError(AXE_INVALID_VARIABLE);

   if (program == NULL)
   {
      free_variable(variable);
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   }

   if (variable->ID == NULL)
   {
      free_variable(variable);
      notifyError(AXE_VARIABLE_ID_UNSPECIFIED);
   }

   if (variable->type == UNKNOWN_TYPE)
   {
      free_variable(variable);
      notifyError(AXE_INVALID_TYPE);
   }

   if (variable->isArray)
   {
      if (variable->arraySize <= 0)
      {
         free_variable(variable);
         notifyError(AXE_INVALID_ARRAY_SIZE);
      }
   }
   
   if (variable->labelID == NULL)
   {
      free_variable(variable);
      notifyError(AXE_INVALID_LABEL);
   }
   
   /* we have to test if already exists a variable with the same ID */
   variableFound = getVariable(program, variable->ID);

   if (variableFound != NULL)
   {
      free_variable(variable);
      notifyError(AXE_VARIABLE_ALREADY_DECLARED);
   }

   /* now we can add the new variable to the program */
   program->variables = addElement(program->variables, variable, -1);

   /* create an instance of `t_axe_data' */
   if (variable->type == INTEGER_TYPE)
   {
      if (variable->isArray)
      {
         new_data_info = alloc_data
               (DIR_SPACE, (variable->arraySize * 4), variable->labelID);
         
         if (new_data_info == NULL)
            notifyError(AXE_OUT_OF_MEMORY);
      }
      else
      {
         new_data_info = alloc_data
            (DIR_WORD, variable->init_val, variable->labelID);
         
         if (new_data_info == NULL)
            notifyError(AXE_OUT_OF_MEMORY);
      }
   }

   /* update the list of directives */
   program->data = addElement(program->data, new_data_info, -1);

   /* update the content of the symbol table */
   sy_error = putSym(program->sy_table, variable->ID, variable->type);
     
   if (sy_error != SY_TABLE_OK)
      notifyError(AXE_SY_TABLE_ERROR);
}

int getNewRegister(t_program_infos *program)
{
   int result;
   
   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   result = program->current_register;
   program->current_register++;
   
   /* return the current label identifier */
   return result;
}

void finalizeProgramInfos(t_program_infos *program)
{
   if (program == NULL)
      return;
   if (program->variables != NULL)
      finalizeVariables(program->variables);
   if (program->instructions != NULL)
      finalizeInstructions(program->instructions);
   if (program->data != NULL)
      finalizeDataSegment(program->data);
   if (program->lmanager != NULL)
      finalize_label_manager(program->lmanager);
   if (program->sy_table != NULL)
      finalize_sy_table(program->sy_table);

   _AXE_FREE_FUNCTION(program);
}

void writeAssembly(t_program_infos *program, char *output_file)
{
   FILE *fp;
   int _error;

   /* test the preconditions */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* If necessary, set the value of `output_file' to "output.asm" */
   if (output_file == NULL)
   {
      /* set "output.o" as output file name */
      output_file = "output.asm";
   }

#ifndef NDEBUG
   fprintf(stdout, "\n\n*******************************************\n");
   fprintf(stdout, "INITIALIZING OUTPUT FILE: %s. \n", output_file);
   fprintf(stdout, "CODE SEGMENT has a size of %d instructions \n"
         , getLength(program->instructions));
   fprintf(stdout, "DATA SEGMENT has a size of %d elements \n"
         , getLength(program->data));
   fprintf(stdout, "NUMBER OF LABELS : %d. \n"
         , get_number_of_labels(program->lmanager));
   fprintf(stdout, "*******************************************\n\n");
#endif
   
   /* open a new file */
   fp = fopen(output_file, "w");
   if (fp == NULL)
      notifyError(AXE_FOPEN_ERROR);

   /* print the data segment */
   translateDataSegment(program, fp);

   /* print the code segment */
   translateCodeSegment(program, fp);

   /* close the file and return */
   _error = fclose(fp);
   if (_error == EOF)
      notifyError(AXE_FCLOSE_ERROR);
}

void printOpcode(int opcode, FILE *fp)
{
   char *opcode_to_string;
   int _error;
   
   /* preconditions: fp must be different from NULL */
   if (fp == NULL)
      notifyError(AXE_INVALID_INPUT_FILE);

   switch(opcode)
   {
      case ADD : opcode_to_string = "add"; break;
	case ADDU : opcode_to_string = "addu"; break;
	case AND : opcode_to_string = "and"; break;
	case DIV : opcode_to_string = "div"; break;
	case DIVU : opcode_to_string = "divu"; break;
	case MUL : opcode_to_string = "mul"; break;
	case MULO : opcode_to_string = "mulo"; break;
	case MULOU : opcode_to_string = "mulou"; break;
	case NOR : opcode_to_string = "nor"; break;
	case OR : opcode_to_string = "or"; break;
	case REM : opcode_to_string = "rem"; break;
	case REMU : opcode_to_string = "remu"; break;
	case SLLV : opcode_to_string = "sllv"; break;
	case SRAV : opcode_to_string = "srav"; break;
	case SRLV : opcode_to_string = "srlv"; break;
	case ROL : opcode_to_string = "rol"; break;
	case ROR : opcode_to_string = "ror"; break;
	case SUB : opcode_to_string = "sub"; break;
	case SUBU : opcode_to_string = "subu"; break;
	case XOR : opcode_to_string = "xor"; break;
	case SLT : opcode_to_string = "slt"; break;
	case SLTU : opcode_to_string = "sltu"; break;
	case SEQ : opcode_to_string = "seq"; break;
	case SGE : opcode_to_string = "sge"; break;
	case SGEU : opcode_to_string = "sgeu"; break;
	case SGT : opcode_to_string = "sgt"; break;
	case SGTU : opcode_to_string = "sgtu"; break;
	case SLE : opcode_to_string = "sle"; break;
	case SLEU : opcode_to_string = "sleu"; break;
	case SNE : opcode_to_string = "sne"; break;
	case MOVN : opcode_to_string = "movn"; break;
	case MOVZ : opcode_to_string = "movz"; break;
	case ADDI : opcode_to_string = "addi"; break;
	case ADDIU : opcode_to_string = "addiu"; break;
	case ANDI : opcode_to_string = "andi"; break;
	case ORI : opcode_to_string = "ori"; break;
	case SLL : opcode_to_string = "sll"; break;
	case SRA : opcode_to_string = "sra"; break;
	case SRL : opcode_to_string = "srl"; break;
	case XORI : opcode_to_string = "xori"; break;
	case SLTI : opcode_to_string = "slti"; break;
	case SLTIU : opcode_to_string = "sltiu"; break;
	case ABS : opcode_to_string = "abs"; break;
	case CLO : opcode_to_string = "clo"; break;
	case CLZ : opcode_to_string = "clz"; break;
	case DV : opcode_to_string = "dv"; break;
	case DVU : opcode_to_string = "dvu"; break;
	case MULT : opcode_to_string = "mult"; break;
	case MULTU : opcode_to_string = "multu"; break;
	case MADD : opcode_to_string = "madd"; break;
	case MADDU : opcode_to_string = "maddu"; break;
	case MSUB : opcode_to_string = "msub"; break;
	case NEG : opcode_to_string = "neg"; break;
	case NEGU : opcode_to_string = "negu"; break;
	case NOTB : opcode_to_string = "not"; break;
	case JALR : opcode_to_string = "jalr"; break;
	case TEQ : opcode_to_string = "teq"; break;
	case TNE : opcode_to_string = "tne"; break;
	case TGE : opcode_to_string = "tge"; break;
	case TGEU : opcode_to_string = "tgeu"; break;
	case TLT : opcode_to_string = "tlt"; break;
	case TLTU : opcode_to_string = "tltu"; break;
	case MOVE : opcode_to_string = "move"; break;
	case LUI : opcode_to_string = "lui"; break;
	case LI	 : opcode_to_string = "li"; break;
	case TEQI : opcode_to_string = "teqi"; break;
	case TNEI : opcode_to_string = "tnei"; break;
	case TGEI : opcode_to_string = "tgei"; break;
	case TGEIU : opcode_to_string = "tgeiu"; break;
	case TLTI : opcode_to_string = "tlti"; break;
	case TLTIU : opcode_to_string = "tltiu"; break;
	case JR : opcode_to_string = "jr"; break;
	case MFHI : opcode_to_string = "mfhi"; break;
	case MFLO : opcode_to_string = "mflo"; break;
	case MTHI : opcode_to_string = "mthi"; break;
	case MTLO : opcode_to_string = "mtlo"; break;
	case LA : opcode_to_string = "la"; break;
	case LB : opcode_to_string = "lb"; break;
	case LBU : opcode_to_string = "lbu"; break;
	case LH : opcode_to_string = "lh"; break;
	case LHU : opcode_to_string = "lhu"; break;
	case LW : opcode_to_string = "lw"; break;
	case LWC1 : opcode_to_string = "lwc1"; break;
	case LWL : opcode_to_string = "lwl"; break;
	case LWR : opcode_to_string = "lwr"; break;
	case LD : opcode_to_string = "ld"; break;
	case ULH : opcode_to_string = "ulh"; break;
	case ULHU : opcode_to_string = "ulhu"; break;
	case ULW : opcode_to_string = "ulw"; break;
	case SB : opcode_to_string = "sb"; break;
	case SH : opcode_to_string = "sh"; break;
	case SW : opcode_to_string = "sw"; break;
	case SWC1 : opcode_to_string = "swc1"; break;
	case SDC1 : opcode_to_string = "sdc1"; break;
	case SWL : opcode_to_string = "swl"; break;
	case SWR : opcode_to_string = "swr"; break;
	case SD : opcode_to_string = "sd"; break;
	case USH : opcode_to_string = "ush"; break;
	case USW : opcode_to_string = "usw"; break;
	case BEQ : opcode_to_string = "beq"; break;
	case BNE : opcode_to_string = "bne"; break;
	case BGE : opcode_to_string = "bge"; break;
	case BGEU : opcode_to_string = "bgeu"; break;
	case BGT : opcode_to_string = "bgt"; break;
	case BGTU : opcode_to_string = "bgtu"; break;
	case BLE : opcode_to_string = "ble"; break;
	case BLEU : opcode_to_string = "bleu"; break;
	case BLT : opcode_to_string = "blt"; break;
	case BLTU : opcode_to_string = "bltu"; break;
	case BNEZ : opcode_to_string = "bnez"; break;
	case BGEZ : opcode_to_string = "bgez"; break;
	case BGEZAL : opcode_to_string = "bgezal"; break;
	case BGTZ : opcode_to_string = "bgtz"; break;
	case BLEZ : opcode_to_string = "blez"; break;
	case BLTZAL : opcode_to_string = "bltzal"; break;
	case BLTZ : opcode_to_string = "bltz"; break;
	case BEQZ : opcode_to_string = "beqz"; break;
	case B : opcode_to_string = "b"; break;
	case J : opcode_to_string = "j"; break;
	case JAL : opcode_to_string = "jal"; break;
	case NOP : opcode_to_string = "nop"; break;
	case SYSCALL : opcode_to_string = "syscall"; break;
      default :
         /* close the file and return */
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_INVALID_OPCODE);
   }
      
   /* postconditions */
   if (fprintf(fp, "%s", opcode_to_string) < 0)
   {
      _error = fclose(fp);
      if (_error == EOF)
         notifyError(AXE_FCLOSE_ERROR);
      notifyError(AXE_FWRITE_ERROR);
   }
}

void printRegister(t_axe_register *reg, FILE *fp)
{
   int _error;
   char *register_name;
   
   /* preconditions: fp must be different from NULL */
   if (fp == NULL)
      notifyError(AXE_INVALID_INPUT_FILE);
   if (reg == NULL)
   {
      _error = fclose(fp);
      if (_error == EOF)
         notifyError(AXE_FCLOSE_ERROR);
      notifyError(AXE_INVALID_REGISTER_INFO);
   }
   if (reg->ID == REG_INVALID)
   {
      _error = fclose(fp);
      if (_error == EOF)
         notifyError(AXE_FCLOSE_ERROR);
      notifyError(AXE_INVALID_REGISTER_INFO);
   }
   switch(reg->ID){
	   case REG_0 : register_name = "$zero"; break;
	   case REG_AT : register_name = "$at"; break;
	   case REG_V0 : register_name = "$v0"; break;
	   case REG_V1 : register_name = "$v1"; break;
	   case REG_A0 : register_name = "$a0"; break;
	   case REG_A1 : register_name = "$a1"; break;
	   case REG_A2 : register_name = "$a2"; break;
	   case REG_A3 : register_name = "$a3"; break;
	   case 0 : register_name = "$t0"; break;
	   case 1 : register_name = "$t1"; break;
	   case 2 : register_name = "$t2"; break;
	   case 3 : register_name = "$t3"; break;
	   case 4 : register_name = "$t4"; break;
	   case 5 : register_name = "$t5"; break;
	   case 6 : register_name = "$t6"; break;
	   case 7 : register_name = "$t7"; break;
	   case 8 : register_name = "$s0"; break;
	   case 9 : register_name = "$s1"; break;
	   case 10 : register_name = "$s2"; break;
	   case 11 : register_name = "$s3"; break;
	   case 12 : register_name = "$s4"; break;
	   case 13 : register_name = "$s5"; break;
	   case 14 : register_name = "$s6"; break;
	   case 15 : register_name = "$s7"; break;
	   case 16 : register_name = "$t8"; break;
	   case 17 : register_name = "$t9"; break;
	   case REG_K0 : register_name = "$k0"; break;
	   case REG_K1 : register_name = "$k1"; break;
	   case REG_GP : register_name = "$gp"; break;
	   case REG_SP: register_name = "$sp"; break;
	   case REG_FP : register_name = "$fp"; break;
	   case REG_RA : register_name = "$ra"; break;
	   default : register_name = "$bad_reg";
   }
   
   if (reg->indirect)
   {
      if (fprintf(fp, "(%s)", register_name) < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }
   }
   else
   {
      if (fprintf(fp, "%s", register_name) < 0)
      {
         _error = fclose(fp);
         if (_error == EOF)
            notifyError(AXE_FCLOSE_ERROR);
         notifyError(AXE_FWRITE_ERROR);
      }
   }
}

t_axe_label * getLabelFromVariableID(t_program_infos *program, char *ID)
{
   t_axe_variable *var;
   
   var = getVariable(program, ID);
   if (var == NULL)
      return NULL;

   /* test the postconditions */
   assert(var->labelID != NULL);
   
   return var->labelID;
}

void finalizeVariables(t_list *variables)
{
   t_list *current_element;
   t_axe_variable *current_var;

   if (variables == NULL)
      return;

   /* initialize the `current_element' */
   current_element = variables;
   while(current_element != NULL)
   {
      current_var = (t_axe_variable *) LDATA(current_element);
      if (current_var != NULL)
      {
         if (current_var->ID != NULL)
            free(current_var->ID);
         
         _AXE_FREE_FUNCTION(current_var);
      }
      
      current_element = LNEXT(current_element);
   }

   /* free the list of variables */
   freeList(variables);
}

int getInstructionGroup(int opcode)
{
	int result;
   switch(opcode)
   {
        case ADD : 
	case ADDU : 
	case AND : 
	case DIV :
	case DIVU :
	case MUL :
	case MULO :
	case MULOU : 
	case NOR :
	case OR : 
	case REM : 
	case REMU : 
	case SLLV : 
	case SRAV :
	case SRLV :
	case ROL :
	case ROR :
	case SUB :
	case SUBU :
	case XOR :
	case SLT :
	case SLTU :
	case SEQ :
	case SGE :
	case SGEU :
	case SGT :
	case SGTU :
	case SLE :
	case SLEU :
	case SNE :
	case MOVN : 
	case MOVZ : result = TERN; break;
	case ADDI :
	case ADDIU :
	case ANDI :
	case ORI :
	case SLL :
	case SRA :
	case SRL :
	case XORI :
	case SLTI :
	case SLTIU : result = BIN_IMM; break;
	case ABS :
	case CLO :
	case CLZ :
	case DV :
	case DVU :
	case MULT :
	case MULTU :
	case MADD :
	case MADDU :
	case MSUB :
	case NEG :
	case NEGU :
	case NOTB :
	case JALR : 
	case TEQ : 
	case TNE : 
	case TGE : 
	case TGEU : 
	case TLT :
	case TLTU :
	case MOVE : result = BIN; break;
	case LUI :
	case LI	 :
	case TEQI :
	case TNEI :
	case TGEI :
	case TGEIU :
	case TLTI :
	case TLTIU : result = UNA_IMM; break;
	case JR : 
	case MFHI :
	case MFLO :
	case MTHI :
	case MTLO : result = UNA; break;
	case LA : 
	case LB : 
	case LBU : 
	case LH : 
	case LHU : 
	case LW : 
	case LWC1 : 
	case LWL : 
	case LWR : 
	case LD : 
	case ULH : 
	case ULHU :
	case ULW :
	case SB :
	case SH :
	case SW :
	case SWC1 :
	case SDC1 :
	case SWL :
	case SWR :
	case SD :
	case USH :
	case USW : result = LOAD_STORE; break;
	case BEQ :
	case BNE :
	case BGE :
	case BGEU :
	case BGT :
	case BGTU :
	case BLE :
	case BLEU :
	case BLT :
	case BLTU :
	case BNEZ : result = BIN_JUMP; break;
	case BGEZ :
	case BGEZAL : 
	case BGTZ :
	case BLEZ :
	case BLTZAL :
	case BLTZ :
	case BEQZ : result = UNA_JUMP; break;
	case B :
	case J : 
	case JAL : result = UNCOND_JUMP; break;
	case NOP :
	case SYSCALL : result = SPECIAL; break;
        default : result = INVALID_GROUP;
   }
   return result;
}
