/*
 * Andrea Di Biagio
 * Politecnico di Milano, 2007
 * 
 * axe_gencode.c
 * Formal Languages & Compilers Machine, 2007/2008
 * 
 */

#include "axe_gencode.h"
#include "axe_errors.h"

static t_axe_instruction * gen_ternary_instruction (t_program_infos *program
	, int opcode, int r_dest, int r_source1, int r_source2);
static t_axe_instruction * gen_binary_instruction (t_program_infos *program
	, int opcode, int r_dest, int r_source1);
static t_axe_instruction * gen_binary_immediate_instruction (t_program_infos *program
	, int opcode, int r_dest, int r_source1, int immediate);
static t_axe_instruction * gen_unary_immediate_instruction(t_program_infos *program
		, int opcode, int r_dest, int immediate);
static t_axe_instruction * gen_unary_instruction(t_program_infos *program
	, int opcode, int r_source1);
static t_axe_instruction * gen_bin_jump_instruction (t_program_infos *program
	, int opcode, int r_source1, int r_source2, t_axe_label *label, int addr);
static t_axe_instruction * gen_unary_jump_instruction (t_program_infos *program
	, int opcode, int r_source1, t_axe_label *label, int addr);
static t_axe_instruction * gen_uncond_jump_instruction (t_program_infos *program
      , int opcode, t_axe_label *label, int addr);
static t_axe_instruction * gen_special_instruction(t_program_infos *program
	, int opcode);
/*static t_axe_instruction * gen_load_store_instruction (t_program_infos *program
	, int opcode, int r_dest, int r_source1, t_axe_label *label, int immediate
	, int addressing_mode);*/
	
 /*-------------------------------------------------------
  *                 TERNARY INSTRUCTIONS
  *------------------------------------------------------*/

/* ADD	Addition with overflow */
t_axe_instruction * gen_add_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, ADD, r_dest, r_source1, r_source2);
}

/* ADDU	Addition without overflow */
t_axe_instruction * gen_addu_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, ADDU, r_dest, r_source1, r_source2);
}
/* AND	and bitwise */
t_axe_instruction * gen_and_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, AND, r_dest, r_source1, r_source2);
}

/* DIV	Divide with overflow (pseudoinstruction) */
t_axe_instruction * gen_div_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, DIV, r_dest, r_source1, r_source2);
}

/* DIVU	Divide without overflow immediate (pseudoinstruction) */
t_axe_instruction * gen_divu_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, DIVU, r_dest, r_source1, r_source2);
}

/* MUL	Multiply without overflow */
t_axe_instruction * gen_mul_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, MUL, r_dest, r_source1, r_source2);
}

/* MULO	Multiply with overflow (pseudoinstruction) */
t_axe_instruction * gen_mulo_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, MULO, r_dest, r_source1, r_source2);
}

/* MULOU	Unsigned multiply with overflow (pseudoinstruction) */
t_axe_instruction * gen_mulou_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, MULOU, r_dest, r_source1, r_source2);
}

/* NOR	bitwise nor */
t_axe_instruction * gen_nor_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, NOR, r_dest, r_source1, r_source2);
}

/* OR	bitwise or*/
t_axe_instruction * gen_or_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, OR, r_dest, r_source1, r_source2);
}

/* REM	Remainder (pseudoinstruction) */
t_axe_instruction * gen_rem_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, REM, r_dest, r_source1, r_source2);
}

/* REMU	Unsigned remainder (pseudoinstruction) */
t_axe_instruction * gen_remu_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, REMU, r_dest, r_source1, r_source2);
}

/* SLLV	Shift left logical variable */
t_axe_instruction * gen_sllv_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SLLV, r_dest, r_source1, r_source2);
}

/* SRAV	Shift right arithmetic variable */
t_axe_instruction * gen_srav_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SRAV, r_dest, r_source1, r_source2);
}

/* SRLV	Shift right logical variable */
t_axe_instruction * gen_srlv_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SRLV, r_dest, r_source1, r_source2);
}

/* ROL	Rotate Left (pseudoinstruction) */
t_axe_instruction * gen_rol_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, ROL, r_dest, r_source1, r_source2);
}

/* ROR	Rotate Right (pseudoinstruction) */
t_axe_instruction * gen_ror_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, ROR, r_dest, r_source1, r_source2);
}

/* SUB	Subtract with overflow */
t_axe_instruction * gen_sub_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SUB, r_dest, r_source1, r_source2);
}

/* SUBU	Subtract without overflow */
t_axe_instruction * gen_subu_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SUBU, r_dest, r_source1, r_source2);
}

/* XOR	Exclusive or */
t_axe_instruction * gen_xor_instruction (t_program_infos *program
      , int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, XOR, r_dest, r_source1, r_source2);
}

/* SLT	Set less than */
t_axe_instruction * gen_slt_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SLT, r_dest, r_source1, r_source2);
}

/* SLTU	Set less than unsigned */
t_axe_instruction * gen_sltu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SLTU, r_dest, r_source1, r_source2);
}

/* SEQ	Set equal (pseudoinstruction) */
t_axe_instruction * gen_seq_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SEQ, r_dest, r_source1, r_source2);
}

/* SGE	Set greater than equal (pseudoinstruction) */
t_axe_instruction * gen_sge_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SGE, r_dest, r_source1, r_source2);
}

/* SGEU	Set greater than equal unsigned (pseudoinstruction) */
t_axe_instruction * gen_sgeu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SGEU, r_dest, r_source1, r_source2);
}

/* SGT	Set greater than (pseudoinstruction) */
t_axe_instruction * gen_sgt_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SGT, r_dest, r_source1, r_source2);
}

/* SGTU	Set greater than unsigned (pseudoinstruction) */
t_axe_instruction * gen_sgtu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SGTU, r_dest, r_source1, r_source2);
}

/* SLE	Set less than equal (pseudoinstruction) */
t_axe_instruction * gen_sle_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SLE, r_dest, r_source1, r_source2);
}

/* SLEU	Set less than equal unsigned (pseudoinstruction) */
t_axe_instruction * gen_sleu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SLEU, r_dest, r_source1, r_source2);
}

/*  SNE	Set not equal (pseudoinstruction) */
t_axe_instruction * gen_sne_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, SNE, r_dest, r_source1, r_source2);
}

/*  MOVN	Move conditional not zero */
t_axe_instruction * gen_movn_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, MOVN, r_dest, r_source1, r_source2);
}

/*  MOVZ	Move conditional zero */
t_axe_instruction * gen_movz_instruction(t_program_infos *program
	, int r_dest, int r_source1, int r_source2)
{
	return gen_ternary_instruction (program, MOVZ, r_dest, r_source1, r_source2);
}

 /*-------------------------------------------------------
  *            BINARY IMMEDIATE INSTRUCTIONS
  *------------------------------------------------------*/

/* ADDI 	Addition immediate with overflow */
t_axe_instruction * gen_addi_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, ADDI, r_dest, r_source1, immediate);
}

/* ADDIU	Addition immediate without overflow */
t_axe_instruction * gen_addiu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, ADDIU, r_dest, r_source1, immediate);
}

/* ANDI		bitwise and immediate */
t_axe_instruction * gen_andi_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, ANDI, r_dest, r_source1, immediate);
}
	
/* ORI		 bitwise or immediate */
t_axe_instruction * gen_ori_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, ORI, r_dest, r_source1, immediate);
}

/* SLL	Shift left logical */
t_axe_instruction * gen_sll_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, SLL, r_dest, r_source1, immediate);
}

/* SRA	Shift rigth arithmetic */
t_axe_instruction * gen_sra_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, SRA, r_dest, r_source1, immediate);
}

/* SRL	Shift rigth logical */
t_axe_instruction * gen_srl_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, SRL, r_dest, r_source1, immediate);
}

/* XORI	xor immediate */
t_axe_instruction * gen_xori_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, XORI, r_dest, r_source1, immediate);
}

/* SLTI	Set less than immediate */
t_axe_instruction * gen_slti_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, SLTI, r_dest, r_source1, immediate);
}

/* SLTIU	Set less than unsigned immediate */
t_axe_instruction * gen_sltiu_instruction(t_program_infos *program
	, int r_dest, int r_source1, int immediate)
{
	return gen_binary_immediate_instruction (program, SLTIU, r_dest, r_source1, immediate);
}

/*-------------------------------------------------------
 *                 BINARY INSTRUCTIONS
 *------------------------------------------------------*/

/* ABS	Absolute Value */
t_axe_instruction * gen_abs_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, ABS, r_dest, r_source1);
}

/* CLO	Count leading ones */
t_axe_instruction * gen_clo_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, CLO, r_dest, r_source1);
}

/* CLZ	Count leading zeros */
t_axe_instruction * gen_clz_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, CLZ, r_dest, r_source1);
}

/* DV	Divide with overflow (quotient in register lo and the remainder in register hi) */
t_axe_instruction * gen_dv_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, DV, r_dest, r_source1);
}

/* DVU	Divide without overflow immediate (quotient in register lo and the remainder in register hi) */
t_axe_instruction * gen_dvu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, DVU, r_dest, r_source1);
}

/* MULT	Signed Multiply (low-order word in lo and high-order word in hi) */
t_axe_instruction * gen_mult_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MULT, r_dest, r_source1);
}

/* MULTU	Unsigned Multiply (low-order word in lo and high-order word in hi) */
t_axe_instruction * gen_multu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MULTU, r_dest, r_source1);
}

/* MADD	Multiply add (64 bit result in the concatenated register lo and hi) */
t_axe_instruction * gen_madd_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MADD, r_dest, r_source1);
}

/* MADDU	Unsigned multiply add (64 bit result in the concatenated register lo and hi) */
t_axe_instruction * gen_maddu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MADDU, r_dest, r_source1);
}

/* MSUB	Multiply subtract (64 bit result in the concatenated register lo and hi) */
t_axe_instruction * gen_msub_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MSUB, r_dest, r_source1);
}

/* NEG	Negate value with overflow (pseudoinstruction) */
t_axe_instruction * gen_neg_instruction (t_program_infos *program
      , int r_dest, int r_source1)
{
	return gen_binary_instruction (program, NEG, r_dest, r_source1);
}

/* NEGU	Negate value without overflow (pseudoinstruction) */
t_axe_instruction * gen_negu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, NEGU, r_dest, r_source1);
}

/* NOT	bitwise negation (pseudoinstruction) */
t_axe_instruction * gen_not_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, NOTB, r_dest, r_source1);
}

/* JALR	Jump and link register */
t_axe_instruction * gen_jalr_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, JALR, r_dest, r_source1);
}

/* TEQ	Trap if equal */
t_axe_instruction * gen_teq_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TEQ, r_dest, r_source1);
}

/* TNE	Trap if not equal */
t_axe_instruction * gen_tne_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TNE, r_dest, r_source1);
}

/* TGE	Trap if greater equal */
t_axe_instruction * gen_tge_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TGE, r_dest, r_source1);
}

/* TGEU	Unigned Trap if greater equal */
t_axe_instruction * gen_tgeu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TGEU, r_dest, r_source1);
}

/* TLT	Trap if less than */
t_axe_instruction * gen_tlt_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TLT, r_dest, r_source1);
}

/* TLTU	Unsigned Trap if less than */
t_axe_instruction * gen_tltu_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, TLTU, r_dest, r_source1);
}

/* MOVE	Move */
t_axe_instruction * gen_move_instruction(t_program_infos *program
	, int r_dest, int r_source1)
{
	return gen_binary_instruction (program, MOVE, r_dest, r_source1);
}

/*-------------------------------------------------------
 *              UNARY IMMEDIATE INSTRUCTIONS
 *------------------------------------------------------*/

/* LUI	Load upper immediate */
t_axe_instruction * gen_lui_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, LUI, r_dest, immediate);
}

/* LI	Load immediate (pseudoinstruction) */
t_axe_instruction * gen_li_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, LI, r_dest, immediate);
}	

/* TEQI	Trap if equal immediate */
t_axe_instruction * gen_teqi_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TEQI, r_dest, immediate);
}	

/* TNEI	Trap if not equal immediate */
t_axe_instruction * gen_tnei_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TNEI, r_dest, immediate);
}	

/* TGEI	Trap if greater equal immediate */
t_axe_instruction * gen_tgei_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TGEI, r_dest, immediate);
}	

/* TGEIU	Unsigned trap if greater equal immediate */
t_axe_instruction * gen_tgeiu_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TGEIU, r_dest, immediate);
}	

/* TLTI	Trap if less than immediate */
t_axe_instruction * gen_tlti_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TLTI, r_dest, immediate);
}	

/* TLTIU	Unsigned Trap if less than immediate */
t_axe_instruction * gen_tltiu_instruction(t_program_infos *program
	, int r_dest, int immediate)
{
	gen_unary_immediate_instruction(program, TLTIU, r_dest, immediate);
}	

/*-------------------------------------------------------
 *                 UNARY INSTRUCTIONS
 *------------------------------------------------------*/


/* JR	Jump register */
t_axe_instruction * gen_jr_instruction(t_program_infos *program
	, int r_source1)
{
	gen_unary_instruction(program, JR, r_source1);
}

/* MFHI	Move from hi */
t_axe_instruction * gen_mfhi_instruction(t_program_infos *program
	, int r_dest)
{
	gen_unary_instruction(program, MFHI, r_dest);
}

/* MFLO	Move from lo */
t_axe_instruction * gen_mflo_instruction(t_program_infos *program
	, int r_dest)
{
	gen_unary_instruction(program, MFLO, r_dest);
}

/* MTHI	Move to hi */
t_axe_instruction * gen_mthi_instruction(t_program_infos *program
	, int r_source1)
{
	gen_unary_instruction(program, MTHI, r_source1);
}	

/* MTLO	Move to lo */
t_axe_instruction * gen_mtlo_instruction(t_program_infos *program
	, int r_source1)
{
	gen_unary_instruction(program, MTLO, r_source1);
}	

/*-------------------------------------------------------
 *              LOAD/STORE INSTRUCTIONS
 *------------------------------------------------------*/

/* LA	Load Address */
t_axe_instruction * gen_la_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LA, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LB	Load Byte */
 t_axe_instruction * gen_lb_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LB, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LBU	Load unsigned byte */
 t_axe_instruction * gen_lbu_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LBU, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LH	Load Halfword */
 t_axe_instruction * gen_lh_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LH, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LHU	Load unsigned halfword */
 t_axe_instruction * gen_lhu_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LHU, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LW	Load word */
 t_axe_instruction * gen_lw_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LW, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LWC1	Load word coprocessor 1 */
 t_axe_instruction * gen_lwc1_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LWC1, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LWL	Load word left */
 t_axe_instruction * gen_lwl_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LWL, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LWR	Load word right */
 t_axe_instruction * gen_lwr_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LWR, r_dest, r_source1, label, immediate, addressing_mode);
}

/* LD	Load doubleword (pseudoinstruction)  */
 t_axe_instruction * gen_ld_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, LD, r_dest, r_source1, label, immediate, addressing_mode);
}

/* ULH	Unaligned load halfword (pseudoinstruction) */
 t_axe_instruction * gen_ulh_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, ULH, r_dest, r_source1, label, immediate, addressing_mode);
}

/* ULHU	Unaligned load halfword unsigned (pseudoinstruction) */
 t_axe_instruction * gen_ulhu_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, ULHU, r_dest, r_source1, label, immediate, addressing_mode);
}

/* ULW	Unaligned load word (pseudoinstruction) */
 t_axe_instruction * gen_ulw_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, ULW, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SB	Store Byte */
 t_axe_instruction * gen_sb_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SB, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SH	Store Halfword */
 t_axe_instruction * gen_sh_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SH, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SW	Store word */
 t_axe_instruction * gen_sw_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SW, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SWC1	Store word coprocessor 1 */
 t_axe_instruction * gen_swc1_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SWC1, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SDC1	Store double coprocessor 1 */
 t_axe_instruction * gen_sdc1_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SDC1, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SWL	Store word left */
 t_axe_instruction * gen_swl_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SWL, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SWR	Store word right */
 t_axe_instruction * gen_swr_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SWR, r_dest, r_source1, label, immediate, addressing_mode);
}

/* SD	Store doubleword (pseudoinstruction) */
 t_axe_instruction * gen_sd_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, SD, r_dest, r_source1, label, immediate, addressing_mode);
}

/* USH	Unaligned Store halfword (pseudoinstruction) */
 t_axe_instruction * gen_ush_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, USH, r_dest, r_source1, label, immediate, addressing_mode);
}

/* USW	Unaligned Store word (pseudoinstruction) */
 t_axe_instruction * gen_usw_instruction(t_program_infos *program
	, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	gen_load_store_instruction (program, USW, r_dest, r_source1, label, immediate, addressing_mode);
}

 /*-------------------------------------------------------
 *              BINARY JUMP INSTRUCTIONS
 *------------------------------------------------------*/

/* BEQ	Branch on equal */
t_axe_instruction * gen_beq_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BEQ, r_source1, r_source2, label, addr);
}
	
/* BNE	Branch on not equal */
t_axe_instruction * gen_bne_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BNE, r_source1, r_source2, label, addr);
}

/* BGE	Branch on greater than equal (pseudoinstruction) */
t_axe_instruction * gen_bge_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BGE, r_source1, r_source2, label, addr);
}

/* BGEU	Branch on greater than equal unsigned (pseudoinstruction) */		
t_axe_instruction * gen_bgeu_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BGEU, r_source1, r_source2, label, addr);
}

/* BGT	Branch on greater than (pseudoinstruction) */
t_axe_instruction * gen_bgt_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BGT, r_source1, r_source2, label, addr);
}

/* BGTU	Branch on greater than unsigned (pseudoinstruction) */
t_axe_instruction * gen_bgtu_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BGTU, r_source1, r_source2, label, addr);
}

/* BLE	Branch on less than equal (pseudoinstruction) */
t_axe_instruction * gen_ble_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BLE, r_source1, r_source2, label, addr);
}

/* BLEU	Branch on less than equal unsigned (pseudoinstruction) */
t_axe_instruction * gen_bleu_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BLEU, r_source1, r_source2, label, addr);
}

/* BLT	Branch on less than (pseudoinstruction) */
t_axe_instruction * gen_blt_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BLT, r_source1, r_source2, label, addr);
}

/* BLTU	Branch on less than unsigned (pseudoinstruction) */		
t_axe_instruction * gen_bltu_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BLTU, r_source1, r_source2, label, addr);
}

/* BNEZ	Branch on not equal zero (pseudoinstruction) */		
t_axe_instruction * gen_bnez_instruction(t_program_infos *program
	, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	gen_bin_jump_instruction (program, BNEZ, r_source1, r_source2, label, addr);
}

/*-------------------------------------------------------
 *              UNARY JUMP INSTRUCTIONS
 *------------------------------------------------------*/

/* BGEZ	Branch on greater than equal zero */		
t_axe_instruction * gen_bgez_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BGEZ, r_source1, label, addr);
}

/* BGEZAL	Branch on greater than equal zero and link */		
t_axe_instruction * gen_bgezal_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BGEZAL, r_source1, label, addr);
}

/* BGTZ	Brach on greater than zero */
t_axe_instruction * gen_bgtz_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BGTZ, r_source1, label, addr);
}

/* BLEZ	Branch on less than equal zero */		
t_axe_instruction * gen_blez_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BLEZ, r_source1, label, addr);
}

/* BLTZAL	Branch on less than and link */		
t_axe_instruction * gen_bltzal_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BLTZAL, r_source1, label, addr);
}

/* BLTZ	Branch on less than zero */
t_axe_instruction * gen_bltz_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BLTZ, r_source1, label, addr);
}

/* BEQZ	Branch on equal zero (pseudoinstruction) */		
t_axe_instruction * gen_beqz_instruction(t_program_infos *program
	, int r_source1, t_axe_label *label, int addr)
{
	gen_unary_jump_instruction(program, BEQZ, r_source1, label, addr);
}

/*-------------------------------------------------------
 *              UNCONDITIONALLY JUMP INSTRUCTIONS
 *------------------------------------------------------*/

/* B	Unconditionally Branch */
t_axe_instruction * gen_b_instruction(t_program_infos *program
	, t_axe_label *label, int addr)
{
	return gen_uncond_jump_instruction (program, B, label, addr);
}

/* J	Unconditionally */
t_axe_instruction * gen_j_instruction(t_program_infos *program
	,t_axe_label *label, int addr)
{
	return gen_uncond_jump_instruction (program, J, label, addr);
}

/* JAL	Jump and link */
t_axe_instruction * gen_jal_instruction(t_program_infos *program
	,t_axe_label *label, int addr)
{
	return gen_uncond_jump_instruction (program, JAL, label, addr);
}

/*-------------------------------------------------------
 *              SPECIAL INSTRUCTIONS
 *------------------------------------------------------*/

/* NOP */
t_axe_instruction * gen_nop_instruction(t_program_infos *program)
{
	return gen_special_instruction(program, NOP);
}

/* SYSCALL */
t_axe_instruction * gen_syscall_instruction(t_program_infos *program)
{
	fprintf(stdout, "syscall \n");
	return gen_special_instruction(program, SYSCALL);
}

/* HALT */
t_axe_instruction * gen_halt_instruction(t_program_infos *program)
{
   t_axe_instruction *instr;
   t_axe_label *label; //ricardo
   t_axe_address *address;
   int addressType; //ricardo
   
   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   
  
   label = assignNewLabel(program);
   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(B,AM_DIRECT);
   
   /* asign a label `t_axe_label' to */
   instr->labelID = label;
   //instr->user_comment = (char *)"HALT";

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);
   
   addressType = LABEL_TYPE;
   
   address = alloc_address(addressType, 0, label);
   
   if (address == NULL)
   {
	   free_Instruction(instr);
	   notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the instruction address info */
   instr->address = address;

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}

/*GENERATE TERNARY INSTRUCTION*/
t_axe_instruction * gen_ternary_instruction (t_program_infos *program
      , int opcode, int r_dest, int r_source1, int r_source2)
{
   t_axe_instruction *instr;
   t_axe_register *reg_dest;
   t_axe_register *reg_source1;
   t_axe_register *reg_source2;
   
   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* test if value is correctly initialized */
   if (  (r_dest == REG_INVALID)
         || (r_source1 == REG_INVALID)
         || (r_source2 == REG_INVALID))
   {
      notifyError(AXE_INVALID_REGISTER_INFO);
   }

   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(opcode,AM_DIRECT);

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* initialize a register info */
   reg_dest = alloc_register(r_dest, 0 );
   if (reg_dest == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the reg_1 info */
   instr->reg_1 = reg_dest;

   reg_source1 = alloc_register(r_source1, 0);
   if (reg_source1 == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the reg_2 info */
   instr->reg_2 = reg_source1;

   reg_source2 = alloc_register(r_source2, 0);
   if (reg_source1 == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the reg_3 info */
   instr->reg_3 = reg_source2;

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}

/*GENERATE BINARY INSTRUCTION*/
t_axe_instruction * gen_binary_immediate_instruction (t_program_infos *program
      , int opcode, int r_dest, int r_source1, int immediate)
{
   t_axe_instruction *instr;
   t_axe_register *reg_dest;
   t_axe_register *reg_source1;

   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* test if value is correctly initialized */
   if (  (r_dest == REG_INVALID)
         || (r_source1 == REG_INVALID) )
   {
      notifyError(AXE_INVALID_REGISTER_INFO);
   }

   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(opcode,AM_DIRECT);

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* initialize a register info */
   reg_dest = alloc_register(r_dest, 0);
   if (reg_dest == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the reg_1 info */
   instr->reg_1 = reg_dest;

   reg_source1 = alloc_register(r_source1, 0);
   if (reg_source1 == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
      return NULL;
   }

   /* update the reg_1 info */
   instr->reg_2 = reg_source1;

   /* assign an immediate value */
   instr->immediate = immediate;

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}

/*GENERATE BINARY INSTRUCTION*/
t_axe_instruction * gen_binary_instruction (t_program_infos *program
      , int opcode, int r_dest, int r_source1)
{
   t_axe_instruction *instr;
   t_axe_register *reg_dest;
   t_axe_register *reg_source1;

   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* test if value is correctly initialized */
   if (  (r_dest == REG_INVALID)
         || (r_source1 == REG_INVALID) )
   {
      notifyError(AXE_INVALID_REGISTER_INFO);
   }

   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(opcode,AM_DIRECT);

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* initialize a register info */
   reg_dest = alloc_register(r_dest, 0);
   if (reg_dest == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the reg_1 info */
   instr->reg_1 = reg_dest;

   reg_source1 = alloc_register(r_source1, 0);
   if (reg_source1 == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
      return NULL;
   }

   /* update the reg_1 info */
   instr->reg_2 = reg_source1;

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}


/* Generate Unary Immediate Instruction */
t_axe_instruction * gen_unary_immediate_instruction(t_program_infos *program
		, int opcode, int r_dest, int immediate)
{
	
	t_axe_instruction *instr;
	t_axe_register *reg_dest;

	/* test if program is initialized */
	if (program == NULL)
		notifyError(AXE_PROGRAM_NOT_INITIALIZED);

	/* test if value is correctly initialized */
	if (r_dest == REG_INVALID)
	{
		notifyError(AXE_INVALID_REGISTER_INFO);
	}
	
	/* test if the opcode is a valid opcode */
	if (opcode == INVALID_OPCODE)
		notifyError(AXE_INVALID_OPCODE);

	/* create an instance of `t_axe_instruction' */
	instr = alloc_instruction(opcode,AM_DIRECT);

	if (instr == NULL)
		notifyError(AXE_OUT_OF_MEMORY);

	/* initialize a register info */
	reg_dest = alloc_register(r_dest, 0);
	
	if (reg_dest == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the reg_1 info */
	instr->reg_1 = reg_dest;

	/* assign an immediate value */
	instr->immediate = immediate;

	/* add the newly created instruction to the current program */
	addInstruction(program, instr);

	/* return the instruction */
	return instr;
}

/* Generate Unary Instruction */
t_axe_instruction * gen_unary_instruction(t_program_infos *program
	, int opcode, int r_source1)
{
	t_axe_instruction *instr;
	t_axe_register *reg;
	
	/* test if program is initialized */
	if (program == NULL)
		notifyError(AXE_PROGRAM_NOT_INITIALIZED);
	
	if (r_source1 == REG_INVALID)
		notifyError(AXE_INVALID_REGISTER_INFO);

	/* test if the opcode is a valid opcode */
	if (opcode == INVALID_OPCODE)
		notifyError(AXE_INVALID_OPCODE);
	
	/* create an instance of `t_axe_instruction' */
	instr = alloc_instruction(opcode,AM_DIRECT);

	if (instr == NULL)
		notifyError(AXE_OUT_OF_MEMORY);

	/* initialize a register info */
	reg = alloc_register(r_source1, 0);

	if (reg == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}
   
	/* update the reg_1 info */
	instr->reg_1 = reg;
	
	/* add the newly created instruction to the current program */
	addInstruction(program, instr);

	/* return the load instruction */
	return instr;
}


/* Generate load/Store Instruction */
/*
ADDRESSING MODES
#define AM_INVALID			-1
#define AM_DIRECT			0
#define AM_INDIRECT			1
#define AM_IMMEDIATE			2
#define AM_INDIRECT_IMMEDIATE		3
#define AM_LABEL			4
#define AM_LABEL_IMMEDIATE		5
#define AM_INDIRECT_LABEL_ADDRESS	6
#define AM_INDIRECT_LABEL_IMMEDIATE	7
*/
t_axe_instruction * gen_load_store_instruction (t_program_infos *program
         , int opcode, int r_dest, int r_source1, t_axe_label *label, int immediate, int addressing_mode)
{
	t_axe_instruction *instr;
	t_axe_register *reg_dest;
	t_axe_register *reg_source1;
	t_axe_address *address;
	int addressType;

	/* test if program is initialized */
	if (program == NULL)
		notifyError(AXE_PROGRAM_NOT_INITIALIZED);
	
	if (r_dest == REG_INVALID)
		notifyError(AXE_INVALID_REGISTER_INFO);
	
	/* test if the opcode is a valid opcode */
	if (opcode == INVALID_OPCODE)
		notifyError(AXE_INVALID_OPCODE);
	
	/* test if the addressing_mode is a valid addressing mode */
	if (addressing_mode == AM_INVALID)
		notifyError(AXE_INVALID_AM);
	
	/* create an instance of `t_axe_instruction' */
	instr = alloc_instruction(opcode,addressing_mode);
	
	if (instr == NULL)
		notifyError(AXE_OUT_OF_MEMORY);
	
	/* initialize a register info */
	reg_dest = alloc_register(r_dest, 0);
	
	if (reg_dest == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}
	
	/* update the reg_1 info */
	instr->reg_1 = reg_dest;
	
	if( (addressing_mode == AM_INDIRECT) ||
		(addressing_mode == AM_INDIRECT_IMMEDIATE) ||
		(addressing_mode == AM_INDIRECT_LABEL_IMMEDIATE)||
		(addressing_mode == AM_INDIRECT_LABEL_ADDRESS)
	  )
	{
		if (r_source1 == REG_INVALID)
			notifyError(AXE_INVALID_REGISTER_INFO);
		/* initialize a register info */
		reg_source1 = alloc_register(r_dest, CG_INDIRECT_SOURCE);
		
		if (reg_source1 == NULL)
		{
			free_Instruction(instr);
			notifyError(AXE_OUT_OF_MEMORY);
		}
		/* update the reg_1 info */
		instr->reg_2 = reg_source1;
		
		if(addressing_mode == AM_INDIRECT)
		{
			addInstruction(program, instr);
			return instr;
		}
	}
	
	if((addressing_mode == AM_IMMEDIATE)||
	  (addressing_mode == AM_INDIRECT_IMMEDIATE)||
	  (addressing_mode == AM_INDIRECT_LABEL_IMMEDIATE)||
	  (addressing_mode == AM_LABEL_IMMEDIATE))
	{
		instr->immediate = immediate;
		
		if((addressing_mode != AM_INDIRECT_LABEL_IMMEDIATE)&&
		  (addressing_mode != AM_LABEL_IMMEDIATE)
		  )
		{
			addInstruction(program, instr);
			return instr;
		}
	}
	/* test if value is correctly initialized */
	if (label != NULL)
	{
		if (label->labelID == LABEL_UNSPECIFIED)
		{
			notifyError(AXE_INVALID_LABEL);
		}
		
		/* address type is a label type */
		addressType = LABEL_TYPE;
		
		/* initialize an address info */
		address = alloc_address(LABEL_TYPE, 0, label);
		
		if (address == NULL)
		{
			free_Instruction(instr);
			notifyError(AXE_OUT_OF_MEMORY);
		}
			/* update the instruction address info */
		instr->address = address;
	
			/* add the newly created instruction to the current program */
		addInstruction(program, instr);
	
		/* return the load instruction */
		return instr;
	}
	else{
		free_Instruction(instr);
		notifyError(AXE_INVALID_LABEL);
	}
	
	return instr;
}


/* Generate Special Instruction */
t_axe_instruction * gen_special_instruction(t_program_infos *program
	, int opcode)
{
   t_axe_instruction *instr;

   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);
   
   /* test if the opcode is a valid opcode */
   if (opcode == INVALID_OPCODE)
      notifyError(AXE_INVALID_OPCODE);
   
   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(opcode,AM_DIRECT);

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}

/* GENERATE BINARY JUMP INSTRUCTION */
t_axe_instruction * gen_bin_jump_instruction (t_program_infos *program
		, int opcode, int r_source1, int r_source2, t_axe_label *label, int addr)
{
	t_axe_instruction *instr;
	t_axe_address * address;
	t_axe_register *reg_source1;
	t_axe_register *reg_source2;
	int addressType;

	/* test if program is initialized */
	if (program == NULL)
		notifyError(AXE_PROGRAM_NOT_INITIALIZED);

	/* test if value is correctly initialized */
	if ( (      r_source1 == REG_INVALID)
	        || (r_source2 == REG_INVALID))
	{
		notifyError(AXE_INVALID_REGISTER_INFO);
	}
	
	/* test if value is correctly initialized */
	if (label != NULL)
	{
		if (label->labelID == LABEL_UNSPECIFIED)
			notifyError(AXE_INVALID_LABEL);

		addressType = LABEL_TYPE;
	}
	else
	{
		if (addr < 0)
			notifyError(AXE_INVALID_ADDRESS);

		addressType = ADDRESS_TYPE;
	}

	/* test if the opcode is a valid opcode */
	if (opcode == INVALID_OPCODE)
		notifyError(AXE_INVALID_OPCODE);

	/* create an instance of `t_axe_instruction' */
	instr = alloc_instruction(opcode,AM_DIRECT);

	if (instr == NULL)
		notifyError(AXE_OUT_OF_MEMORY);

	/* initialize an address info */
	address = alloc_address(addressType, addr, label);
   
	if (address == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the instruction address info */
	instr->address = address;
	
	reg_source1 = alloc_register(r_source1, 0);
	if (reg_source1 == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the reg_2 info */
	instr->reg_1 = reg_source1;

	reg_source2 = alloc_register(r_source2, 0);
	if (reg_source1 == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the reg_3 info */
	instr->reg_2 = reg_source2;

	/* add the newly created instruction to the current program */
	addInstruction(program, instr);

	/* return the load instruction */
	return instr;
}

/* GENERATE UNARY JUMP INSTRUCTION */
t_axe_instruction * gen_unary_jump_instruction (t_program_infos *program
		, int opcode, int r_source1, t_axe_label *label, int addr)
{
	t_axe_instruction *instr;
	t_axe_address * address;
	t_axe_register *reg_source1;
	int addressType;

	/* test if program is initialized */
	if (program == NULL)
		notifyError(AXE_PROGRAM_NOT_INITIALIZED);

	/* test if value is correctly initialized */
	if ( r_source1 == REG_INVALID)
	{
		notifyError(AXE_INVALID_REGISTER_INFO);
	}
	
	/* test if value is correctly initialized */
	if (label != NULL)
	{
		if (label->labelID == LABEL_UNSPECIFIED)
			notifyError(AXE_INVALID_LABEL);

		addressType = LABEL_TYPE;
	}
	else
	{
		if (addr < 0)
			notifyError(AXE_INVALID_ADDRESS);

		addressType = ADDRESS_TYPE;
	}

	/* test if the opcode is a valid opcode */
	if (opcode == INVALID_OPCODE)
		notifyError(AXE_INVALID_OPCODE);

	/* create an instance of `t_axe_instruction' */
	instr = alloc_instruction(opcode,AM_DIRECT);

	if (instr == NULL)
		notifyError(AXE_OUT_OF_MEMORY);

	/* initialize an address info */
	address = alloc_address(addressType, addr, label);
   
	if (address == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the instruction address info */
	instr->address = address;
	
	reg_source1 = alloc_register(r_source1, 0);
	if (reg_source1 == NULL)
	{
		free_Instruction(instr);
		notifyError(AXE_OUT_OF_MEMORY);
	}

	/* update the reg_2 info */
	instr->reg_1 = reg_source1;

	/* add the newly created instruction to the current program */
	addInstruction(program, instr);

	/* return the load instruction */
	return instr;
}


/* GENERATE UNCONDITIONALLY JUMP INSTRUCTION */
t_axe_instruction * gen_uncond_jump_instruction (t_program_infos *program
      , int opcode, t_axe_label *label, int addr)
{
   t_axe_instruction *instr;
   t_axe_address * address;
   int addressType;

   /* test if program is initialized */
   if (program == NULL)
      notifyError(AXE_PROGRAM_NOT_INITIALIZED);

   /* test if value is correctly initialized */
   if (label != NULL)
   {
      if (label->labelID == LABEL_UNSPECIFIED)
         notifyError(AXE_INVALID_LABEL);

      addressType = LABEL_TYPE;
   }
   else
   {
      if (addr < 0)
         notifyError(AXE_INVALID_ADDRESS);

      addressType = ADDRESS_TYPE;
   }

   /* test if the opcode is a valid opcode */
   if (opcode == INVALID_OPCODE)
      notifyError(AXE_INVALID_OPCODE);

   /* create an instance of `t_axe_instruction' */
   instr = alloc_instruction(opcode,AM_DIRECT);

   if (instr == NULL)
      notifyError(AXE_OUT_OF_MEMORY);

   /* initialize an address info */
   address = alloc_address(addressType, addr, label);
   
   if (address == NULL)
   {
      free_Instruction(instr);
      notifyError(AXE_OUT_OF_MEMORY);
   }

   /* update the instruction address info */
   instr->address = address;

   /* add the newly created instruction to the current program */
   addInstruction(program, instr);

   /* return the load instruction */
   return instr;
}


